// Copyright (c) Microsoft Corporation. All rights reserved.

#include <XuiEffect10.fxh>

struct VSOUT
{
    float4 Position : SV_POSITION;
    float2 uvTex    : TEXCOORD0; 
    float2 uvShadow : TEXCOORD1;
};

struct PSOUT
{
    float4 Color : SV_TARGET;
};

RasterizerState rsCullNONE  { CullMode = None; };

VSOUT ShadeVertex(const float2 Position0 : POSITION0)
{
    VSOUT Output;

    float4 pos; // position in screen space
    pos = mul(float4(Position0.x, Position0.y, 0, 1), transpose(g_mWorldViewProj));
    pos.z = g_vParams.x;

    float4 posClip; // position in clip space
    posClip = mul(pos, transpose(g_mInvViewport));
    Output.Position = posClip;

    float4 uv = mul(float4((Position0.x), (Position0.y), 0, 1), transpose(g_mLocalToTexTransform));
    Output.uvTex.x = uv.x + g_vUVTrans.x;
    Output.uvTex.y = uv.y + g_vUVTrans.y;
    Output.uvShadow = Output.uvTex;

    return Output;
}

VSOUT ShadeTextVertex(const float2 Position0 : POSITION0, const float2 uv : TEXCOORD0)
{
    VSOUT Output;

    float4 pos; // position in screen space
    pos = mul(float4(Position0.x, Position0.y, 0, 1), transpose(g_mWorldViewProj));
    pos.x += .5;
    pos.y += .5;
    pos.z = g_vParams.x;

    float4 posClip; // position in clip space
    posClip = mul(pos, transpose(g_mInvViewport));
    Output.Position = posClip;

    Output.uvTex = uv; // texture has already been calculated
    Output.uvShadow.x = uv.x - g_vShadowOffset.x;
    Output.uvShadow.y = uv.y - g_vShadowOffset.y;

    return Output;
}

PSOUT ShadeSolidPixel(VSOUT Input)
{
    PSOUT output = (PSOUT) 0;

    output.Color = g_vSolidColor * g_vColorFactor;

    return output;
}

PSOUT ShadeTexPixel(VSOUT Input)
{
    PSOUT output = (PSOUT) 0;

    output.Color = txBase.Sample(samTextureSampler, Input.uvTex) * g_vColorFactor;

    return output;
}

PSOUT ShadeTextDropShadowPixel(VSOUT Input)
{
    PSOUT output = (PSOUT) 0;

    // sample the texture for the glyph and the dropshadow
    float4 vfColorGlyph = txBase.Sample(samTextureSampler, Input.uvTex) * g_vColorFactor;    
    float4 vfColorShadow = txBase.Sample(samTextureSampler, Input.uvShadow) * g_vShadowColor;

    // blend the shadow color based on how strong the glyph color is here 
    vfColorShadow *= (1 - vfColorGlyph.a);
    vfColorGlyph *= (1 - vfColorShadow.a);
    
    // combine the results for the final coloring.
    output.Color = vfColorGlyph + vfColorShadow;

    return output;
}

PSOUT ShadeRadialPixel(VSOUT Input)
{
    PSOUT output = (PSOUT) 0;

    float fdist = length(Input.uvTex)*2;
    float2 uv = float2(fdist, 0);
    output.Color = txBase.Sample(samTextureSampler, uv) * g_vColorFactor;

    return output;
}

technique10 T0
{
    pass P0
    {
      #ifdef SHD_VERTEX
        SetVertexShader(CompileShader(vs_4_0, ShadeVertex()));
      #endif
      #ifdef SHD_TEXTVERTEX
        SetVertexShader(CompileShader(vs_4_0, ShadeTextVertex()));
      #endif

      SetGeometryShader(NULL);
    
      #ifdef SHD_SOLIDPIXEL
        SetPixelShader(CompileShader(ps_4_0, ShadeSolidPixel()));
      #endif
      #ifdef SHD_TEXPIXEL
        SetPixelShader(CompileShader(ps_4_0, ShadeTexPixel()));
      #endif
      #ifdef SHD_RADIALPIXEL
        SetPixelShader(CompileShader(ps_4_0, ShadeRadialPixel()));
      #endif
      #ifdef SHD_TEXTDROPSHADOWPIXEL
        SetPixelShader(CompileShader(ps_4_0, ShadeTextDropShadowPixel()));
      #endif

      SetRasterizerState(rsCullNONE);
    }
}

