const bool EffectDeclaration
<
    string Name      = "Xui";
    string Class     = "Basic";
    string Fallback  = "Null";
    bool   NewMaterialUsage = true;
> = true;

texture BaseTexture          : MATERIAL_BASE_TEXTURE;

dword State_TextureWrapU     : STATE_TEXTUREWRAPU = 1;
dword State_TextureWrapV     : STATE_TEXTUREWRAPV = 1;
dword State_MinFilter        : STATE_MINFILTER = 2;
dword State_MagFilter        : STATE_MAGFILTER = 2;
dword State_MipFilter        : STATE_MIPFILTER = 2;

bool  State_AlphaBlendEnable : STATE_ALPHABLENDENABLE = true;
bool State_AlphaTestEnable   : STATE_ALPHATESTENABLE = false;
dword State_SrcBlend         : STATE_SRCBLEND = 1;
dword State_DstBlend         : STATE_DSTBLEND = 2;
bool State_FogEnable         : STATE_FOGENABLE = false;

// VS Vars
dword State_CullMode         : STATE_CULLMODE = 1;	// No culling by default
float4x4 InvViewport         : SHADERMATRIXPARAM0;	// inverse of viewport scaling matrix
float4x4 WorldViewProj       : SHADERMATRIXPARAM1;	// matWorldViewProj
float4 Params                : SHADERVECTORPARAM0;	// (0, .25, 2, 4)
float2 ShadowOffset          : SHADERVECTORPARAM1;	// textcoord offsets for dropshadow 
float4 UVTrans               : SHADERVECTORPARAM2;	// UV translation
float4x4 LocalToTexTransform : SHADERMATRIXPARAM2;	// from local coordinate system to texture uv transformation matrix

// PS Vars
float4 SolidColor  : SHADERVECTORPARAM3;
float4 ColorFactor : SHADERVECTORPARAM4;			// color to modulate with
float4 ShadowColor : SHADERVECTORPARAM5;			// color for dropshadow


sampler TextureSampler = sampler_state
{
    Texture       = (BaseTexture);
    AddressU      = (State_TextureWrapU);
    AddressV      = (State_TextureWrapV);
    MinFilter     = (State_MinFilter);
    MagFilter     = (State_MagFilter);
    MipFilter     = (State_MipFilter);
};

struct VSOUT
{
    float4 Position : POSITION;
    float2 uvTex : TEXCOORD0; 
    float2 uvShadow : TEXCOORD1;
};

VSOUT ShadeVertex(const float2 Position0 : POSITION0)
{
    VSOUT Output;
    float4 pos; // position in screen space
    pos = mul(float4(Position0.x, Position0.y, 0, 1), transpose(WorldViewProj));
    pos.x -= .5;
    pos.y -= .5;
    pos.z = Params.x;

    float4 posClip; // position in clip space
    posClip = mul(pos, transpose(InvViewport));
    Output.Position = posClip;

    float4 uv = mul(float4((Position0.x), (Position0.y), 0, 1), transpose(LocalToTexTransform));
    Output.uvTex.x = uv.x + UVTrans.x;
    Output.uvTex.y = uv.y + UVTrans.y;
    Output.uvShadow = Output.uvTex;

    return Output;
}

VSOUT ShadeTextVertex(const float2 Position0 : POSITION0, const float2 uv : TEXCOORD0)
{
    VSOUT Output;

    float4 pos; // position in screen space
    pos = mul(float4(Position0.x, Position0.y, 0, 1), transpose(WorldViewProj));
    pos.z = Params.x;

    float4 posClip; // position in clip space
    posClip = mul(pos, transpose(InvViewport));
    Output.Position = posClip;

    Output.uvTex = uv;                 // texture has already been calculated
    Output.uvShadow.x = uv.x - ShadowOffset.x;
    Output.uvShadow.y = uv.y - ShadowOffset.y;

    return Output;
}

float4 ShadeSolidPixel(VSOUT Input) : COLOR
{
    float4 output = SolidColor*ColorFactor;
    return output;
}

float4 ShadeTexPixel(VSOUT Input) : COLOR
{
    float4 output = tex2D(TextureSampler, Input.uvTex)*ColorFactor;
    return output;
}

float4 ShadeTextDropShadowPixel(VSOUT Input) : COLOR
{
    // sample the texture for the glyph and the dropshadow
    float4 vfColorGlyph = tex2D(TextureSampler, Input.uvTex)*ColorFactor;    
    float4 vfColorShadow = tex2D(TextureSampler, Input.uvShadow)*ShadowColor;

    // blend the shadow color based on how strong the glyph color is here 
    vfColorShadow *= (1 - vfColorGlyph.a);
    vfColorGlyph *= (1 - vfColorShadow.a);
    
    // combine the results for the final coloring.
    float4 output = vfColorGlyph + vfColorShadow;
    return output;
}

float4 ShadeRadialPixel(VSOUT Input) : COLOR
{
    float fdist = length(Input.uvTex)*2;
    float2 uv = float2(fdist, 0);
    float4 output = tex2D(TextureSampler, uv)*ColorFactor;
    return output;
}

technique ShaderEffect
{
    pass p0
    {
        CullMode         = (State_CullMode);
        AlphaBlendEnable = (State_AlphaBlendEnable);
        AlphaTestEnable  = (State_AlphaTestEnable);
        SrcBlend         = (State_SrcBlend);
        DestBlend        = (State_DstBlend);
        FogEnable        = (State_FogEnable);

    #ifdef SHD_VERTEX
        VertexShader = compile vs_2_0 ShadeVertex();
    #endif
    #ifdef SHD_TEXTVERTEX
        VertexShader = compile vs_2_0 ShadeTextVertex();
    #endif
    
    #ifdef SHD_SOLIDPIXEL
        PixelShader = compile ps_2_0 ShadeSolidPixel();
    #endif
    #ifdef SHD_TEXPIXEL
        PixelShader = compile ps_2_0 ShadeTexPixel();
    #endif
    #ifdef SHD_RADIALPIXEL
        PixelShader = compile ps_2_0 ShadeRadialPixel();
    #endif
    #ifdef SHD_TEXTDROPSHADOWPIXEL
        PixelShader = compile ps_2_0 ShadeTextDropShadowPixel();
    #endif
    }
}

