//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// SwarmCloud40.fx
//

#include "SwarmCloud40.fxh"
#include "FuncLibrary.fxh"

//
// REVIEW: This somewhat twisted set of input parameters matches an existing
// vertex type - basically I didn't want to add a new one just for this purpose
// then have to plumb it all the way through G2D. I might go back and do that
// once I look at sprites.
//

struct VS_INPUT
{
    float4 spriteCenter : POSITION0;
    float3 groupCenter  : NORMAL;
    float4 packedUV     : COLOR0;
    float2 packedCorner : TEXCOORD0;
};

struct VS_OUTPUT
{
    float4 position     : SV_POSITION;
    float4 diffuse      : COLOR;
    float2 uv           : TEXCOORD;
    float  fFogDistance : FOG;
};

VS_OUTPUT VS(VS_INPUT In)
{
    VS_OUTPUT Out;
    
    //    
    // Make a viewpoint oriented billboard matrix
    //
    // See SwarmCloud.cpp, ConstructFacingMatrix() - KEEP SHADER IN SYNC!
    //
    
    float4x4 mBillboard;
    
    float4 scaledSpriteCenter = In.spriteCenter;
    scaledSpriteCenter.x *= g_mWorld[0][0];
    scaledSpriteCenter.y *= g_mWorld[1][1];
    scaledSpriteCenter.z *= g_mWorld[2][2];
    
    mBillboard[1] = float4(0,1,0,0);
    mBillboard[2] = normalize(g_mWorld[3] + scaledSpriteCenter);
    mBillboard[0] = normalize(float4(cross(mBillboard[1], mBillboard[2]), 0));
    mBillboard[1] = float4(cross(mBillboard[2], mBillboard[0]), 0);
    mBillboard[3] = In.spriteCenter;
    
    //
    // Get the corner position relative to the cloud center
    //
    
    float4 corner = float4(In.packedCorner.x, In.packedCorner.y, 0, 1);
    corner = mul(corner, mBillboard);

    //
    // Get the transformed corner position
    //

    Out.position = mul(corner, g_mWorldViewProjection);
    
    //
    // Sun lighting
    //
    // See bglfp.cpp, g2d_LightCloudFromNormal() - KEEP SHADER IN SYNC!
    //
    
    float3 cloudGroupNormal = normalize(corner - In.groupCenter);
    float  fIntensity = dot(g_vSunVectorWorld, cloudGroupNormal);

    if (fIntensity < -g_fMedianLine)
    {
        fIntensity = g_fMinIntensity +
            (g_fMedianIntensity - g_fMinIntensity) *
            ((1 + fIntensity) / (1.0001 - g_fMedianLine));
            // The "1.0001" in the line above is a hack to fix a bug
            // observed on some hardware. Even though the else branch
            // of this if/else was taken a divide by zero in this
            // branch (when g_fMedianLine is 1.0) caused the whole
            // shader to go crazy.
    }
    else
    {
        fIntensity = g_fMedianIntensity +
            (1 - g_fMedianIntensity) *
            ((fIntensity + g_fMedianLine) / (1 + g_fMedianLine));
    }

    float fRed   = fIntensity * g_fCloudDirectional.r + g_fCloudAmbient.r;
    float fGreen = fIntensity * g_fCloudDirectional.g + g_fCloudAmbient.g;
    float fBlue  = fIntensity * g_fCloudDirectional.b + g_fCloudAmbient.b;

    fRed    = min(fRed, 1.0f);
    fGreen  = min(fGreen, 1.0f);
    fBlue   = min(fBlue, 1.0f);
    
    //
    // Height band lighting/coloring
    //
    // See SwarmCloud.cpp, GetColor() - KEEP SHADER IN SYNC!
    //

    float height = corner.y;

    float4 baseColor;
    float4 topColor;
    float  baseHeight;
    float  topHeight;
    
    if (height <= g_rgColorLevelHeights[0])
    {
        baseColor  = g_rgColorLevelColors[0];
        topColor   = g_rgColorLevelColors[0];
        baseHeight = g_rgColorLevelHeights[0];
        topHeight  = g_rgColorLevelHeights[0] + 1; // +1 to avoid division by zero below
    }
    else if (height <= g_rgColorLevelHeights[1])
    {
        baseColor  = g_rgColorLevelColors[0];
        topColor   = g_rgColorLevelColors[1];
        baseHeight = g_rgColorLevelHeights[0];
        topHeight  = g_rgColorLevelHeights[1];
    }
    else if (height <= g_rgColorLevelHeights[2])
    {
        baseColor  = g_rgColorLevelColors[1];
        topColor   = g_rgColorLevelColors[2];
        baseHeight = g_rgColorLevelHeights[1];
        topHeight  = g_rgColorLevelHeights[2];
    }
    else if (height <= g_rgColorLevelHeights[3])
    {
        baseColor  = g_rgColorLevelColors[2];
        topColor   = g_rgColorLevelColors[3];
        baseHeight = g_rgColorLevelHeights[2];
        topHeight  = g_rgColorLevelHeights[3];
    }
    else if (height <= g_rgColorLevelHeights[4])
    {
        baseColor  = g_rgColorLevelColors[3];
        topColor   = g_rgColorLevelColors[4];
        baseHeight = g_rgColorLevelHeights[3];
        topHeight  = g_rgColorLevelHeights[4];
    }
    else
    {
        baseColor  = g_rgColorLevelColors[4];
        topColor   = g_rgColorLevelColors[4];
        baseHeight = g_rgColorLevelHeights[4];
        topHeight  = g_rgColorLevelHeights[4] + 1; // +1 to avoid division by zero below
    }

    float s = (height - baseHeight) / (topHeight - baseHeight);
    float4 color = lerp(baseColor, topColor, s);

    //
    // Fade in/out alpha
    //
    
    float distance_from_center = length(corner);
    
    float fAlpha = 1.0f;
    if (g_fAlpha > 0)
    {
        // Fading in from or out to the edges
        float fMagnitude = 1.3f * g_fRadius - distance_from_center;
        fMagnitude = max(0, fMagnitude);
        fAlpha = g_fAlpha - g_fAlphaEdges * (fMagnitude / g_fRadius);
    }
    else
    {
        // Fading out to the core
        fAlpha = -g_fAlpha - g_fAlphaEdges * (distance_from_center / g_fRadius);
    }
    
    Out.diffuse      = float4(fRed, fGreen, fBlue, saturate(fAlpha)) * color;
    Out.uv.x         = In.packedUV.r;
    Out.uv.y         = In.packedUV.g;
    Out.fFogDistance = FogVS(corner, g_mWorld, g_vEyePoint);

    return Out;
}

struct PS_OUTPUT
{
    float4 cColor : SV_TARGET;
};

PS_OUTPUT PS(VS_OUTPUT In)
{
    PS_OUTPUT Out;
    float4 cColor = In.diffuse * txBase.Sample(samBaseSampler, In.uv);
    Out.cColor = float4(FogPS(cColor.xyz, In.fFogDistance, g_fFogDensity, g_cFog), cColor.a);
    return Out;
}

#define SHD_SWARMCLOUD

#ifdef SHD_SWARMCLOUD

    technique10 T0
    {
        pass P0
        {
            SetVertexShader(CompileShader(vs_4_0, VS()));
            SetGeometryShader(NULL);
            SetPixelShader(CompileShader(ps_4_0, PS()));
        }
    }

#elif SHD_DARK_RAYS

    DARK_RAY_VS_OUTPUT DarkRayVS(VS_INPUT In)
    {
    }

    technique10 T0
    {
        pass P0
        {
            SetVertexShader(CompileShader(vs_4_0, DarkRayVS()));
            SetGeometryShader(NULL);
            SetPixelShader(CompileShader(ps_4_0, DarkRayPS()));
        }
    }

#endif
