//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// SwarmCloud.fx
//

//
// KEEP THE CALCULATIONS IN HERE CONSISTENT WITH
// THE NON-SHADER PATH IN SWARMCLOUD.CPP!
//

#include <Common.fxh>
#include <MaterialDecl.fxh>
#include <D3D9Texture.fxh>
#include <FuncLibrary.fxh>

// Use the same shader constant naming convention as the 4.0 shaders.
#define g_mWorld mWorld
#define g_mWorldViewProj mFullProj
#define g_vSunVectorWorld vSunVectorWorld

const bool EffectDeclaration
<
    string Name     = "SwarmCloud";
    string Class    = "Basic";
    bool   NewMaterialUsage = true;
> = true;

//------------------------------------------------------------------------------
// Swarm cloud parameters
//

//
// REVIEW: There are a lot of parameters here being
// set for every single cloud. Is it worth creating
// some custom parameters and setting them once per
// frame instead?
//

//
// Lighting parameters
//
float   g_fMedianLine           : CLOUDMATERIAL_MEDIANLINE;
float   g_fMinIntensity         : CLOUDMATERIAL_MININTENSITY;
float   g_fMedianIntensity      : CLOUDMATERIAL_MEDIANINTENSITY;
float4  g_fCloudDirectional     : CLOUDMATERIAL_DIRECTIONAL;
float4  g_fCloudAmbient         : CLOUDMATERIAL_AMBIENT;

//
// Fade in/out parameters
//
float   g_fAlpha                : CLOUDMATERIAL_ALPHA;
float   g_fAlphaEdges           : CLOUDMATERIAL_ALPHAEDGES;
float   g_fRadius               : CLOUDMATERIAL_RADIUS;

//
// Level band coloring parameters
//
#define MAX_COLOR_LEVELS 5
float   g_rgColorLevelHeights[MAX_COLOR_LEVELS] : CLOUDMATERIAL_COLORLEVELHEIGHTS;
float4  g_rgColorLevelColors[MAX_COLOR_LEVELS]  : CLOUDMATERIAL_COLORLEVELCOLORS;

//------------------------------------------------------------------------------
// Vertex shader
//

//
// REVIEW: This somewhat twisted set of input parameters
// matches an existing vertex type - basically I didn't
// want to add a new one just for this purpose then have
// to plumb it all the way through G2D. I might go back
// and do that though once I look at sprites.
//

struct VS_INPUT
{
    float4 spriteCenter : POSITION0;
    float3 groupCenter  : NORMAL;
    float4 packedUV     : COLOR0;
    float2 packedCorner : TEXCOORD0;
};

struct VS_OUTPUT
{
    float4 position     : POSITION;
    float4 diffuse      : COLOR;
    float2 uv           : TEXCOORD;
};

VS_OUTPUT VS(VS_INPUT In)
{
    VS_OUTPUT Out;
    
    //    
    // Make a viewpoint oriented billboard matrix
    //
    // See SwarmCloud.cpp, ConstructFacingMatrix() - KEEP SHADER IN SYNC!
    //
    
    float4x4 mBillboard;
    
    float4 scaledSpriteCenter = In.spriteCenter;
    scaledSpriteCenter.x *= g_mWorld[0][0];
    scaledSpriteCenter.y *= g_mWorld[1][1];
    scaledSpriteCenter.z *= g_mWorld[2][2];
    
    mBillboard[1] = float4(0,1,0,0);
    mBillboard[2] = normalize(g_mWorld[3] + scaledSpriteCenter);
    mBillboard[0] = normalize(float4(cross(mBillboard[1], mBillboard[2]), 0));
    mBillboard[1] = float4(cross(mBillboard[2], mBillboard[0]), 0);
    mBillboard[3] = In.spriteCenter;
    
    //
    // Get the corner position relative to the cloud center
    //
    
    float4 corner = float4(In.packedCorner.x, In.packedCorner.y, 0, 1);
    corner = mul(corner, mBillboard);

    //
    // Get the transformed corner position
    //

    Out.position = mul(corner, g_mWorldViewProj);
    
    //
    // Sun lighting
    //
    // See bglfp.cpp, g2d_LightCloudFromNormal() - KEEP SHADER IN SYNC!
    //
    
    float3 cloudGroupNormal = normalize(corner - In.groupCenter);
    float  fIntensity = dot(g_vSunVectorWorld, cloudGroupNormal);

    if (fIntensity < -g_fMedianLine)
    {
        fIntensity = g_fMinIntensity +
            (g_fMedianIntensity - g_fMinIntensity) *
            ((1 + fIntensity) / (1.0001 - g_fMedianLine));
            // The "1.0001" in the line above is a hack to fix a bug
            // observed on some nVidia hardware (a 7800). Even though
            // the else branch of this if/else was taken a divide by
            // zero in this branch (when g_fMedianLine is 1.0) caused
            // the whole shader to go crazy.
    }
    else
    {
        fIntensity = g_fMedianIntensity +
            (1 - g_fMedianIntensity) *
            ((fIntensity + g_fMedianLine) / (1 + g_fMedianLine));
    }

    float fRed   = fIntensity * g_fCloudDirectional.r + g_fCloudAmbient.r;
    float fGreen = fIntensity * g_fCloudDirectional.g + g_fCloudAmbient.g;
    float fBlue  = fIntensity * g_fCloudDirectional.b + g_fCloudAmbient.b;

    fRed    = min(fRed, 1.0f);
    fGreen  = min(fGreen, 1.0f);
    fBlue   = min(fBlue, 1.0f);
    
    //
    // Height band lighting/coloring
    //
    // See SwarmCloud.cpp, GetColor() - KEEP SHADER IN SYNC!
    //

    float height = corner.y;

    float4 baseColor;
    float4 topColor;
    float  baseHeight;
    float  topHeight;
    
    if (height <= g_rgColorLevelHeights[0])
    {
        baseColor  = g_rgColorLevelColors[0];
        topColor   = g_rgColorLevelColors[0];
        baseHeight = g_rgColorLevelHeights[0];
        topHeight  = g_rgColorLevelHeights[0] + 1; // +1 to avoid division by zero below
    }
    else if (height <= g_rgColorLevelHeights[1])
    {
        baseColor  = g_rgColorLevelColors[0];
        topColor   = g_rgColorLevelColors[1];
        baseHeight = g_rgColorLevelHeights[0];
        topHeight  = g_rgColorLevelHeights[1];
    }
    else if (height <= g_rgColorLevelHeights[2])
    {
        baseColor  = g_rgColorLevelColors[1];
        topColor   = g_rgColorLevelColors[2];
        baseHeight = g_rgColorLevelHeights[1];
        topHeight  = g_rgColorLevelHeights[2];
    }
    else if (height <= g_rgColorLevelHeights[3])
    {
        baseColor  = g_rgColorLevelColors[2];
        topColor   = g_rgColorLevelColors[3];
        baseHeight = g_rgColorLevelHeights[2];
        topHeight  = g_rgColorLevelHeights[3];
    }
    else if (height <= g_rgColorLevelHeights[4])
    {
        baseColor  = g_rgColorLevelColors[3];
        topColor   = g_rgColorLevelColors[4];
        baseHeight = g_rgColorLevelHeights[3];
        topHeight  = g_rgColorLevelHeights[4];
    }
    else
    {
        baseColor  = g_rgColorLevelColors[4];
        topColor   = g_rgColorLevelColors[4];
        baseHeight = g_rgColorLevelHeights[4];
        topHeight  = g_rgColorLevelHeights[4] + 1; // +1 to avoid division by zero below
    }

    float s = (height - baseHeight) / (topHeight - baseHeight);
    float4 color = lerp(baseColor, topColor, s);

    //
    // Fade in/out alpha
    //
    
    float distance_from_center = length(corner);
    
    if (g_fAlpha > 0)
    {
        // Fading in from or out to the edges
        float fMagnitude = 1.3f * g_fRadius - distance_from_center;
        fMagnitude = max(0, fMagnitude);
        g_fAlpha = g_fAlpha - g_fAlphaEdges * (fMagnitude / g_fRadius);
    }
    else
    {
        // Fading out to the core
        g_fAlpha = -g_fAlpha - g_fAlphaEdges * (distance_from_center / g_fRadius);
    }
    
    Out.diffuse = float4(fRed, fGreen, fBlue, g_fAlpha) * color;
    Out.uv.x    = In.packedUV.b;
    Out.uv.y    = In.packedUV.g;

    return Out;
}

//------------------------------------------------------------------------------
// Pixel shader
//

float4 PS(VS_OUTPUT In): COLOR
{
    return In.diffuse * tex2D(BaseSampler, In.uv);
}

//------------------------------------------------------------------------------
// Technique
//

technique T0
{
    pass P0
    {
        NormalizeNormals            = TRUE;
        ZWriteEnable                = FALSE;
        ZEnable                     = TRUE;
        ZFunc                       = LESSEQUAL;
        AlphaBlendEnable            = TRUE;
        SrcBlend                    = SRCALPHA;
        DestBlend                   = INVSRCALPHA;
        SeparateAlphaBlendEnable    = FALSE;
        AlphaTestEnable             = FALSE;
        StencilEnable               = FALSE;
        SpecularEnable              = FALSE;
        ColorWriteEnable            = RED | GREEN | BLUE;
        FogEnable                   = TRUE;
        CullMode                    = CCW;
        VertexShader                = compile vs_1_1 VS();
        PixelShader                 = compile ps_1_1 PS();
    }
}
