//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
#include <Common.fxh>

#define EffectPerfLevel_MaxShader20  29

const bool EffectDeclaration
<
    string Name      = "SoftShadow";
    string Class     = "Global";
    string Fallback  = "Null";
> = true;


//-----------------------------------------------------------------------------
// Global constants
//-----------------------------------------------------------------------------
static const int    MAX_SAMPLES            = 16;    // Maximum texture grabs


//-----------------------------------------------------------------------------
// Global variables
//-----------------------------------------------------------------------------

// Contains sampling offsets used by the techniques
float2 g_avSampleOffsets[MAX_SAMPLES];
float4 g_avSampleWeights[MAX_SAMPLES];
float4 g_aFSColor;
float4 g_aFSColorLight;


//-----------------------------------------------------------------------------
// Texture samplers
//-----------------------------------------------------------------------------
sampler s0 : register(s0);

//-----------------------------------------------------------------------------
// Vertex shaders
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Pixel shaders
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Name: FinalScenePass
// Type: Pixel shader                                      
// Desc: Perform blue shift, tone map the scene, and add post-processed light
//       effects
//-----------------------------------------------------------------------------
float4 FinalScenePass
    (
    in float2 vScreenPosition : TEXCOORD0
    ) : COLOR
{
    return (tex2D(s0, vScreenPosition));
}

//-----------------------------------------------------------------------------
// Name: ShadowFSPass
// Type: Pixel shader                                      
// Desc: Performs simple constant color fill
//-----------------------------------------------------------------------------
float4 ShadowFSPass
    (
    ) : COLOR
{
    return g_aFSColor;
}

//-----------------------------------------------------------------------------
// Name: ShadowFSPassLight
// Type: Pixel shader                                      
// Desc: Performs simple constant color fill (for lit volumes)
//-----------------------------------------------------------------------------
float4 ShadowFSPassLight
    (uniform float fAlphaScale) : COLOR
{
    return float4(g_aFSColorLight.rgb,g_aFSColorLight.a/fAlphaScale);
}

float4 ShadowFSPassLightBloom
    (
    ) : COLOR
{
    return float4(1,1,1,g_aFSColorLight.a);
}

float4 ShadowMapClearPS
    (
    ) : COLOR
{
    return float4(99999.9f,99999.9f,99999.9f,1.0f);
}


//-----------------------------------------------------------------------------
// Name: GaussBlur5x5
// Type: Pixel shader                                      
// Desc: Simulate a 5x5 kernel gaussian blur by sampling the 12 points closest
//       to the center point.
//-----------------------------------------------------------------------------
float4 GaussBlur5x5
    (
    in float2 vScreenPosition : TEXCOORD0
    ) : COLOR
{
	
    float4 sample = 0.0f;

	for( int i=0; i < 12; i++ )
	{
		sample += g_avSampleWeights[i] * tex2D( s0, vScreenPosition + g_avSampleOffsets[i] );
	}

	return sample;
}


//-----------------------------------------------------------------------------
// Techniques
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Name: GaussBlur5x5
// Type: Technique                                     
// Desc: Simulate a 5x5 kernel gaussian blur by sampling the 12 points closest
//       to the center point.
//-----------------------------------------------------------------------------
technique GaussBlur5x5
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_2_0 GaussBlur5x5();
    }
}


//-----------------------------------------------------------------------------
// Name: FinalScenePass
// Type: Technique                                     
// Desc: Minimally transform and texture the incoming geometry
//-----------------------------------------------------------------------------
technique FinalScenePass
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_2_0 FinalScenePass();
    }
}

//-----------------------------------------------------------------------------
// Name: ShadowFS
// Type: Technique                                     
// Desc: Minimally transform and texture the incoming geometry
//-----------------------------------------------------------------------------
technique ShadowFS
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_2_0 ShadowFSPass();
    }
}

technique ShadowFSLight
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_2_0 ShadowFSPassLight(2.0f);
    }
}

technique ShadowMapClear
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_2_0 ShadowMapClearPS();
    }
}

