//------------------------------------------------------------------------------
// Copyright (c) Microsoft Corporation.  All Rights Reserved.
//------------------------------------------------------------------------------
#include "Water40.fxh"
#include "Water40Macros.fxh"
#include "Water40Consts.fxh"
#include "FuncLibrary.fxh"

#if defined(SHD_WATER_DEBUG)
#define HDRScale g_fWaterHDRScale
#else
#define HDRScale 1.1f
#endif


struct VS_INPUT
{
    float4 vPos       : POSITION;
    float3 vNormal    : NORMAL;
    float4 vColor     : COLOR;
    float2 vTex       : TEXCOORD0;
};
          
struct VS_OUTPUT
{
    float4  vPos      : SV_POSITION;
    float4  vColor    : COLOR0;

    float4 TexBump1         : TEXCOORD0;
    float4 TexBump2         : TEXCOORD1;
    float4 TexBump3         : TEXCOORD2;
    float4 TexBumpDetail    : TEXCOORD3;

    float4 TexEnvMap  : TEXCOORD10;
    float4 TexBase    : TEXCOORD11;
    float4 MiscData   : TEXCOORD12;
    float3 vNormalWS  : TEXCOORD13;
    float3 vPosWS     : TEXCOORD14;
    float3 vEyePosWS  : TEXCOORD15;
    float4 TexScreen  : TEXCOORD16;
};

float4 ComputeTextureCoordinates(float3 WorldPos, float Rotation, float UVScale, float XSpeed, float YSpeed, float Blend, float WindBlend)
{
    float2 bmp,sc;
    float2 tempBump;
    float waveDirection;

    waveDirection = g_fWindDirection + radians(Rotation);
    bmp = fmod((WorldPos.xz - g_mWorld[3].xz) * (UVScale * 0.0004f),256.0f);
    sc = float2(sin(waveDirection),cos(waveDirection));
    tempBump = bmp + (g_fSimTime * float2(XSpeed,YSpeed) * 0.003f);
    return float4(tempBump.x * sc.y - tempBump.y * sc.x, tempBump.x * sc.x + tempBump.y * sc.y, Blend * WindBlend, Blend);
}

VS_OUTPUT 
VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPos, g_mWorldViewProjection);

    // Transform to world space
    const float3 vWorldPos = mul(v.vPos, g_mWorld);

    // Compute eye vector
    const float3 vEye = normalize(g_vCameraEyePoint - vWorldPos);

    // Compute bumpmap coordinates
    float windFactor = 0.9 + (pow(g_fWindSpeed/32.0f,0.5f) * 0.2f);
    o.TexBump1 = ComputeTextureCoordinates( vWorldPos, GET_BumpRotation1, GET_BumpUVScale1, 
                                            GET_BumpTimeX1, GET_BumpTimeY1,GET_BumpBlend1,windFactor*GET_BumpWind1);
    o.TexBump2 = ComputeTextureCoordinates( vWorldPos, GET_BumpRotation2, GET_BumpUVScale2, 
                                            GET_BumpTimeX2, GET_BumpTimeY2, GET_BumpBlend2,windFactor*GET_BumpWind2);
    o.TexBump3 = ComputeTextureCoordinates( vWorldPos, GET_BumpRotation3, GET_BumpUVScale3, 
                                            GET_BumpTimeX3, GET_BumpTimeY3, GET_BumpBlend3,windFactor*GET_BumpWind3);
    o.TexBumpDetail = ComputeTextureCoordinates( vWorldPos, GET_BumpRotationDetail, 5.0f * GET_BumpUVScaleDetail, 
                                                 GET_BumpTimeXDetail, GET_BumpTimeYDetail, GET_BumpBlendDetail,windFactor*GET_BumpWindDetail);

    // Send out interpolated normal for lighting
    o.vNormalWS = normalize(TransformVectorToSpace(v.vNormal, g_mWorld));

    // Send out eye pos
    o.vPosWS    = vWorldPos;
    o.vEyePosWS = g_vCameraEyePoint;

    // Output vertex color straight to PS
    // The alpha component is used to represent the relative
    // depth of the water, used in computing water translucenty
    // The color component is simply the color scaled by the 
    // diffuse sun lighting component
    // R = Water Class
    // G = Fog
    // B = Intensity
    // A = Alpha
    o.vColor = float4(WaterClass, FogVS(v.vPos, g_mWorld, g_vCameraEyePoint), pow(o.vNormalWS.y,3),v.vColor.a);

    // Environment mapping is determined from the screen-space coordinates
    // of our geometry. This implies that we need to divide them by W, and then
    // offset/scale into a 0-1 range. Because of interpolation issues due to front
    // clipped geometry, the info is pre-processed and passed to the PS where the 1/W
    // is applied. We also apply the normal to the data to emulate reflections caused
    // by the animating geometry
    o.TexEnvMap.xyz = o.vPos.xyz+o.vPos.www;
    o.TexEnvMap.w = 2.0*o.vPos.w;
    o.TexScreen.xyz = float3(1,-1,1)*o.vPos.xyz+o.vPos.www;
    o.TexScreen.w = 2.0*o.vPos.w;

    // Diffuse texture coordinates
    o.TexBase.xy = v.vTex;

#if !defined(SHD_WATER_DEBUG)
    // Compute per-poly fresnel
    o.MiscData =   float4(1.0 - lerp(GET_FresnelFactorMin, GET_FresnelFactorMax, pow(smoothstep(0,1,saturate(-normalize(vWorldPos).y)),0.5)),
                          GET_WhiteCapPower,
                          0,
                          0);
#else
    // Compute per-poly fresnel
    o.MiscData =   float4(1.0 - lerp(GET_FresnelFactorMin, GET_FresnelFactorMax, pow(smoothstep(0,1,saturate(-normalize(vWorldPos).y)),0.5)),
                          GET_WhiteCapPower,
                          (g_iWaterShowClass!=0) ? (g_iWaterShowClass==WaterClass) : 1,
                          0);
#endif

    return o;
}

float4 
PS(VS_OUTPUT Input) : SV_TARGET
{
    float2 BumpEnv;
    float4 Color;

    // Renomalize the input vectors and determine the
    // eye vector for lighting and reflection
    const float3 vEyeVectWS = Input.vEyePosWS - Input.vPosWS;
    const float eyeDist = length(vEyeVectWS);
    const float3 vEyeDirWS = (Input.vEyePosWS - Input.vPosWS) / eyeDist;

    // Determine the detail bump blend (which is based on the eye distance)
    float DetailBumpBlend = pow(saturate(1.0f-(eyeDist/250.0f)),2).x;

    // Sample Detail bump
    float crashFactor = 0.0f;
    float4 tmpBump;
    float4 Bump = float4(0,0,0.25,0);
    float4 BumpDetail = Input.TexBumpDetail.z * normalize((2*txBump[3].Sample(samBump, Input.TexBumpDetail.xy))-1);
    
    // Sample bumpmap 1
    tmpBump = normalize((2*txBump[0].Sample(samBump, Input.TexBump1.xy))-1);
    crashFactor += Input.TexBump1.z * (1-abs(dot(tmpBump,Bump)));
    Bump = normalize(Bump + (Input.TexBump1.z * tmpBump));

    // Sample bumpmap 2
    tmpBump = Input.TexBump2.z * normalize((2*txBump[1].Sample(samBump, Input.TexBump2.xy))-1);
    crashFactor += Input.TexBump2.z * (1-abs(dot(tmpBump,Bump)));
    Bump = normalize(Bump + (Input.TexBump2.z * tmpBump));

    // Sample bumpmap 3
    tmpBump = Input.TexBump3.z * normalize((2*txBump[2].Sample(samBump, Input.TexBump3.xy))-1);
    crashFactor += Input.TexBump3.z * (1-abs(dot(tmpBump,Bump)));
    Bump = normalize(Bump + (Input.TexBump3.z * tmpBump));

    // Compute the basic normal resulting from the bumpmaps
    Bump = normalize( float4(Bump.xy + (DetailBumpBlend*BumpDetail).xy, Bump.z, 0) );
    float3 vSurfaceNorm = Bump.xzy;

    // The crash factor is used to determine the possibility of whitecaps. The assumption (and simplification) 
    // is that whitecaps will occur when various waves (or bumpmaps in our case) colide together. So we take 
    // our higher frequency bumpmaps and do a dot product which gives us a rough idea whether they are in 
    // phase, and this gives us a rough guess...
    crashFactor = pow(Input.MiscData.y * crashFactor,32);
    crashFactor = saturate((0.8f + g_fWindSpeed/32.0f)*crashFactor);
    float4 whiteCap = txBump[4].Sample(samLinear,Input.TexBump1.xy*10.0f);

    // Compute the diffuse lighting component of the water.
    const float lightFactor_sun = saturate(dot(g_vSunVectorWorld_Sun,vSurfaceNorm));
    const float3 vHN = normalize(vEyeDirWS.xyz + g_vSunVectorWorld_Sun.xyz);
    const float3 vHN2 = normalize(vEyeDirWS.xyz + g_vSunVectorWorld_Moon.xyz);

    #if defined(SHD_BLOOM)
    const float2 specularFactor = HDRScale * GET_SpecularBlend *(pow(GET_SpecularBoost*saturate(float2(dot(vSurfaceNorm, vHN),g_fSunMoonInterpolant * dot(vSurfaceNorm, vHN2))), GET_SpecularPower));
    #else
    const float2 specularFactor = GET_SpecularBlend *(pow(GET_SpecularBoost*saturate(float2(dot(vSurfaceNorm, vHN),g_fSunMoonInterpolant * dot(vSurfaceNorm, vHN2))), GET_SpecularPower));
    #endif

    // Sample the environment texture
    BumpEnv = (Input.TexEnvMap.xy/Input.TexEnvMap.w)+ (0.2f*Bump.xy);
    const float4 envColor = txEnvironment.Sample(samClamp, BumpEnv) + pow(crashFactor/2,2).xxxx;

    // Sample the water texture
    const float4 baseColor = txBase.Sample(samClamp, Input.TexBase);
    const float fAlpha = baseColor.a;
        
    // Determine the final color
    Color = float4(( (0.8 * lightFactor_sun+0.6) * lerp(baseColor, (0.5+baseColor)*envColor, Input.MiscData.x)).xyz,fAlpha);
    Color.xyz = lerp(Color.xyz, (0.8 * lightFactor_sun+0.2)*whiteCap.xyz, crashFactor);
    Color = float4(lerp(Color.xyz + (g_vSunDirectional * specularFactor.x) + (0.3 * (specularFactor.y + saturate(dot(g_vSunVectorWorld_Moon,vSurfaceNorm)))).xxx, baseColor.xyz, fAlpha.xxxx),fAlpha);

#if defined(SHD_WATER_DEBUG)
    return lerp(float4(0,0,0,1), float4(FogPS(Color, Input.vColor.g, g_fFogDensity, g_cFog), Color.a), Input.MiscData.z);
#else
    return float4(FogPS(Color, Input.vColor.g, g_fFogDensity, g_cFog), Color.a);
#endif
}

technique10 T0
{
    pass P0
    {
        SetVertexShader(CompileShader(vs_4_0, VS()));
        SetGeometryShader(NULL);
        SetPixelShader(CompileShader(ps_4_0, PS()));
    }
}

