//------------------------------------------------------------------------------
// Copyright (c) Microsoft Corporation.  All Rights Reserved.
//------------------------------------------------------------------------------
#include "Terrain40.fxh"
#include "FuncLibrary.fxh"

// Declare inputs
struct VS_INPUT
{
    float4 vPos : POSITION;
#if !defined (SHDTERRAIN_PRELIT)
    float2 vNormal : NORMAL;
#endif
    float2 vTex : TEXCOORD0;
};


// Declare outputs
struct VS_OUTPUT
{
    float4 vPos  : SV_POSITION;
    float2 TexBase        : TEXCOORD0;

#if !defined (SHDTERRAIN_PRELIT)
    float3 vNormalWS      : TEXCOORD1;
#endif
#if defined(SHDTERRAIN_LANDDETAIL)
    float2 TexLandDetail  : TEXCOORD2;
#endif
#if defined(SHDTERRAIN_WATERDETAIL)
    float2 TexWaterDetail : TEXCOORD3;
#endif
#if defined(SHDTERRAIN_REFLECT)
    float3 vEyeWS         : TEXCOORD4;
#endif

    float  fFogDistance   : FOG;
};

// Code
VS_OUTPUT
VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPos, g_mWorldViewProjection);

    // Base texture coordinates
    o.TexBase.xy = v.vTex;

    // Detail texture coordinates
    #if defined(SHDTERRAIN_LANDDETAIL)
        o.TexLandDetail.xy = v.vTex.xy * g_fLandDetailScale;
    #endif

    #if defined(SHDTERRAIN_WATERDETAIL)
        o.TexWaterDetail.xy = v.vTex.xy * g_fWaterDetailScale;
    #endif

    #if !defined (SHDTERRAIN_PRELIT)
        o.vNormalWS = normalize(mul(v.vNormal, g_mWorld));
    #endif
    
    #if defined(SHDTERRAIN_REFLECT)
    {
        // Transform to world space
        const float3 vWorldPos = mul(v.vPos, g_mWorld);

        // Compute eye vector
        o.vEyeWS = normalize(g_vCameraEyePoint - vWorldPos);
    }
    #endif

    o.fFogDistance = FogVS(v.vPos, g_mWorld, g_vCameraEyePoint);    

    return o;
}

float4 PS(VS_OUTPUT Input) : SV_TARGET
{
    const float4 ColorBase = txBase.Sample(samClamp, Input.TexBase);
           
    #if defined(SHDTERRAIN_LANDDETAIL)
        const float3 LandDetail = txLandDetail.Sample(samWrap, Input.TexLandDetail);
    #endif

    #if defined(SHDTERRAIN_WATERDETAIL)
        const float3 WaterDetail = txWaterDetail.Sample(samWrap, Input.TexWaterDetail);
    #endif
    
    // compute land contribution to final color
    #if defined(SHDTERRAIN_LAND)
        float3 LandColor = ColorBase;
        #if defined(SHDTERRAIN_LANDDETAIL)
            LandColor *= 2 * LandDetail;
        #endif
    #endif

    // compute water contribution to final color
    #if defined(SHDTERRAIN_WATER)
        float3 WaterColor = ColorBase;
        #if defined(SHDTERRAIN_WATERDETAIL)
            WaterColor *= 2 * WaterDetail;
        #endif
    #endif

    // combine land and water colors
    #if defined(SHDTERRAIN_LAND) && defined(SHDTERRAIN_WATER)
        // alpha channel determines visibility of land versus water
        // 255 == land; 0 == water
        const float fLand = ColorBase.a;
        float3 Color = LandColor * fLand + WaterColor * (1 - fLand);
    #elif defined(SHDTERRAIN_LAND)
        float3 Color = LandColor;
    #elif defined(SHDTERRAIN_WATER)
        float3 Color = WaterColor;
    #endif

    // compute lighting
    #if !defined(SHDTERRAIN_PRELIT)
    {
        // Perform diffuse calculation
        const float fDot = saturate(dot(Input.vNormalWS, g_vSunVectorWorld));

        const float3 cSunLight = g_vSunAmbient + (g_vSunDirectional * fDot);
        #if !defined(SHDTERRAIN_HDR)
            cSunLight = saturate(cSunLight);
        #endif
        Color *= cSunLight;
    }
    #endif

    // old-skool water reflection
    #if defined(SHDTERRAIN_REFLECT) && defined(SHDTERRAIN_WATER)
    {
        // Compute normalized reflection vector
        // As we know the normal points straight up, we can simplify this
        // : float3 vReflection = normalize((2 * dot(vEye, vNormal) * vNormal) - vEye);
        const float3 vReflection = normalize(float3(-Input.vEyeWS.x, (2 * Input.vEyeWS.y) - Input.vEyeWS.y, -Input.vEyeWS.z));

        // Tranform by environment matrix
        float2 envCoords;
        envCoords.xy = mul(float4(vReflection.x, vReflection.y, vReflection.z, 1), g_mEnvrionment);

        float3 cEnv = txEnvironment.Sample(samClamp, envCoords);
        
        #if defined(SHDTERRAIN_HDR)
            cEnv *= g_fEmissiveCoefficient;
        #endif

        #if defined(SHDTERRAIN_LAND)
            Color += 0.5 * (cEnv * (1 - fLand));
        #else
            Color += 0.5 * cEnv;
        #endif
    }
    #endif

    Color = FogPS(Color, Input.fFogDistance, g_fFogDensity, g_cFog);
    
    return float4(Color, 1.0);
}

technique10 T0
{
    pass P0
    {
        SetVertexShader(CompileShader(vs_4_0, VS()));
        SetGeometryShader(NULL);
        SetPixelShader(CompileShader(ps_4_0, PS()));
    }
}