//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// Does dynamic lighting and supports HDR
//

#undef SHDTERRAIN_BUMP

#include "Terrain.fxh"

#define vLightDir vSunVectorWorld

const bool EffectDeclaration
<
    string Name      = "Terrain20";
    string Class     = "Terrain";
    string Fallback  = "Assert";
> = true;

// Declare inputs
struct VS_20INPUT
{
    float4 vPos : POSITION;
#if !defined (SHDTERRAIN_PRELIT)
    float2 vNormal : NORMAL;
#endif
    float2 vTex : TEXCOORD0;
};

// Declare outputs
struct VS_20OUTPUT
{
    float4 vPos  : POSITION;
    float2 TexBase        : TEXCOORD0;

#if !defined (SHDTERRAIN_PRELIT)
    float3 vNormalWS      : TEXCOORD1;
#endif
#if defined(SHDTERRAIN_LANDDETAIL)
    float2 TexLandDetail  : TEXCOORD2;
#endif
#if defined(SHDTERRAIN_WATERDETAIL)
    float2 TexWaterDetail : TEXCOORD3;
#endif
#if defined(SHDTERRAIN_REFLECT)
    float3 vEyeWS         : TEXCOORD4;
#endif

    float  fFog     : FOG;
};

// Code
VS_20OUTPUT
Terrain20VS(const VS_20INPUT v)
{
    VS_20OUTPUT o = (VS_20OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPos, mFullProj);

    // Base texture coordinates
    o.TexBase.xy = v.vTex;

    // Detail texture coordinates
    #if defined(SHDTERRAIN_LANDDETAIL)
    {
        o.TexLandDetail.xy = v.vTex.xy * fDetailScaleLand;
    }
    #endif

    #if defined(SHDTERRAIN_WATERDETAIL)
    {
        o.TexWaterDetail.xy = v.vTex.xy * fDetailScaleWater;
    }
    #endif

    #if !defined (SHDTERRAIN_PRELIT)
    {
        o.vNormalWS = normalize(mul(v.vNormal, mWorld));
    }
    #endif
    
    #if defined(SHDTERRAIN_REFLECT)
    {
        // Transform to world space
        float3 vWorldPos = mul(v.vPos, mWorld);

        // Compute eye vector
        o.vEyeWS = normalize(vEyePoint - vWorldPos);
    }
    #endif

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w);

    return o;
}

float4
Terrain20PS(VS_20OUTPUT Input) : COLOR
{
    float4 ColorBase = tex2D(BaseSampler, Input.TexBase);
    
    #if !defined (SHDTERRAIN_PRELIT)
        // Perform diffuse calculation
        float fDot = saturate(dot(Input.vNormalWS, vLightDir));

        float3 cSunLight = vSunAmbient + (vSunDirectional * fDot);
        #if !defined(SHDTERRAIN_HDR)
            cSunLight = saturate(cSunLight);
        #endif
    #endif
        
    #if defined(SHDTERRAIN_LANDDETAIL)
        float3 LandDetail = tex2D(LandDetailSampler, Input.TexLandDetail);
    #endif

    #if defined(SHDTERRAIN_WATERDETAIL)
        float3 WaterDetail = tex2D(WaterDetailSampler, Input.TexWaterDetail);
    #endif

    #if defined(SHDTERRAIN_REFLECT)
        // Compute normalized reflection vector
        // As we know the normal points straight up, we can simplify this
        // : float3 vReflection = normalize((2 * dot(vEye, vNormal) * vNormal) - vEye);
        float3 vReflection = normalize(float3(-Input.vEyeWS.x, (2 * Input.vEyeWS.y) - Input.vEyeWS.y, -Input.vEyeWS.z));

        // Tranform by environment matrix
        float2 envCoords;
        envCoords.xy = mul(float4(vReflection.x, vReflection.y, vReflection.z, 1), mEnv);

        float3 cEnv = tex2D(EnvMapSampler, envCoords);
        #if defined(SHDTERRAIN_HDR)
            cEnv *= g_fEmissiveCoefficient;
        #endif
    #endif

    // compute land contribution to final color
    #if defined(SHDTERRAIN_LAND)
        float3 LandColor = ColorBase;
        #if defined(SHDTERRAIN_LANDDETAIL)
            LandColor *= 2 * LandDetail;
        #endif
    #endif

    // compute water contribution to final color
    #if defined(SHDTERRAIN_WATER)
        float3 WaterColor = ColorBase;
        #if defined(SHDTERRAIN_WATERDETAIL)
            WaterColor *= 2 * WaterDetail;
        #endif
    #endif

    // combine land and water colors
    #if defined(SHDTERRAIN_LAND) && defined(SHDTERRAIN_WATER)
        // alpha channel determines visibility of land versus water
        // 255 == land; 0 == water
        float fLand = ColorBase.a;
        float3 Color = LandColor * fLand + WaterColor * (1 - fLand);
    #elif defined(SHDTERRAIN_LAND)
        float3 Color = LandColor;
    #elif defined(SHDTERRAIN_WATER)
        float3 Color = WaterColor;
    #endif

    // compute lighting
    #if !defined(SHDTERRAIN_PRELIT)
        Color *= cSunLight;
    #endif

    // old-skool water reflection
    #if defined(SHDTERRAIN_REFLECT) && defined(SHDTERRAIN_WATER)
        #if defined(SHDTERRAIN_LAND)
            Color += 0.5 * (cEnv * (1 - fLand));
        #else
            Color += 0.5 * cEnv;
        #endif
    #endif

    return (float4(Color, 0));
}

technique T0
<
>
{
    pass p0
    {
        #include <DefaultStates.fxh>

#ifdef SHD_BLOOM
        FogColor = 0;
#endif
        VertexShader = compile vs_2_0 Terrain20VS();
        PixelShader  = compile ps_2_0 Terrain20PS();
        ColorWriteEnable = Red | Green | Blue | Alpha;
    }
}
