//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
#include <Common.fxh>

// Declare inputs
struct VS_INPUT
{
    float4 vPos : POSITION;
    float2 vTex : TEXCOORD0;
};

// Declare outputs
struct VS_OUTPUT
{
    float4 vPos  : POSITION;
    float  fFog  : FOG;

    float2 TexBase        : TEXCOORD0;
    float2 TexLandDetail  : TEXCOORD1;
    float2 TexWaterDetail : TEXCOORD2;
    float2 TexEnvMap      : TEXCOORD3;
};

struct VSBUMP_OUTPUT
{
    float4 vPos  : POSITION;
    float  fFog  : FOG;

    float2 TexBump        : TEXCOORD0;
    float2 TexEnvMap      : TEXCOORD1;
    float2 TexBase        : TEXCOORD2;
    float2 TexLandDetail  : TEXCOORD3;
};

matrix mEnv      : WATERENVMATRIX;

float fDetailScaleLand : MATERIAL_LAND_DETAIL_SCALE;
float fDetailScaleWater : MATERIAL_WATER_DETAIL_SCALE;
float fBumpScale : MATERIAL_BUMP_SCALE;

// Potential textures
texture BaseTexture        : MATERIAL_BASE_TEXTURE;
texture LandDetailTexture  : MATERIAL_LAND_DETAIL_TEXTURE;
texture WaterDetailTexture : MATERIAL_WATER_DETAIL_TEXTURE;
texture EnvTexture         : MATERIAL_ENVIRONMENT_TEXTURE;
texture BumpTexture        : MATERIAL_BUMP_TEXTURE;

sampler BaseSampler = sampler_state
{
    Texture       = (BaseTexture);
    AddressU      = Clamp;
    AddressV      = Clamp;
    MinFilter     = (State_MinFilter);
    MagFilter     = (State_MagFilter);
    MipFilter     = (State_MipFilter);
    MipMapLodBias = (State_MipMapLodBias);
};

sampler BumpSampler = sampler_state
{
    Texture       = (BumpTexture);
    AddressU      = Wrap;
    AddressV      = Wrap;
    MinFilter     = (State_MinFilter);
    MagFilter     = (State_MagFilter);
    MipFilter     = (State_MipFilter);
    MipMapLodBias = (State_MipMapLodBias);
};

sampler LandDetailSampler = sampler_state
{
    Texture       = (LandDetailTexture);
    AddressU      = Wrap;
    AddressV      = Wrap;
    MinFilter     = (State_MinFilter);
    MagFilter     = (State_MagFilter);
    MipFilter     = (State_MipFilter);
    MipMapLodBias = (State_MipMapLodBias);
};

sampler WaterDetailSampler = sampler_state
{
    Texture       = (WaterDetailTexture);
    AddressU      = Wrap;
    AddressV      = Wrap;
    MinFilter     = (State_MinFilter);
    MagFilter     = (State_MagFilter);
    MipFilter     = (State_MipFilter);
    MipMapLodBias = (State_MipMapLodBias);
};

sampler EnvMapSampler = sampler_state
{
    Texture       = (EnvTexture);
    AddressU      = Clamp;
    AddressV      = Clamp;
    MinFilter     = (State_MinFilter);
    MagFilter     = (State_MagFilter);
    MipFilter     = (State_MipFilter);
    MipMapLodBias = (State_MipMapLodBias);
};

float ComputeFog(const float w)
{
    // Compute vertex fog
    // fog = (fog_end - dist) * (1 / (fog_end-fog_start))
    float fFog = (FOG_END - w) * FOG_RECIP;
    fFog = max(fFog, 0.0);
    fFog = min(fFog, 1.0);

    // Now select whether we're using vertex fog or table fog
    fFog = (fFog * FOG_SELECT_VERTEX) + (w * FOG_SELECT_TABLE);

    return (fFog);
}

// Code
VS_OUTPUT
TerrainVS(const VS_INPUT v,
          uniform bool bLandDetail,
          uniform bool bWaterDetail,
          uniform bool bReflect,
          uniform bool bBump)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPos, mFullProj);

    // Base texture coordinates
    o.TexBase.xy = v.vTex;

    // Detail texture coordinates
    if (bLandDetail)
    {
        o.TexLandDetail.xy = v.vTex.xy * fDetailScaleLand;
    }

    if (bWaterDetail)
    {
        o.TexWaterDetail.xy = v.vTex.xy * fDetailScaleWater;
    }

    if (bReflect || bBump)
    {
        // Transform to world space
        float3 vWorldPos = mul(v.vPos, mWorld);

        // Compute eye vector
        float3 vEye = normalize(vEyePoint - vWorldPos);

        // Compute normalized reflection vector
        // As we know the normal points straight up, we can simplify this
        // : float3 vReflection = normalize((2 * dot(vEye, vNormal) * vNormal) - vEye);
        float3 vReflection = normalize(float3(-vEye.x, (2 * vEye.y) - vEye.y, -vEye.z));

        // Tranform by environment matrix
        o.TexEnvMap.xy = mul(float4(vReflection.x, vReflection.y, vReflection.z, 1), mEnv);
    }

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w);

    return o;
}

VSBUMP_OUTPUT
TerrainBumpVS(const VS_INPUT v,
              uniform bool bLandDetail)
{
    VSBUMP_OUTPUT o = (VS_OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPos, mFullProj);

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w);

    // Transform to world space
    float3 vWorldPos = mul(v.vPos, mWorld);

    // Scale bumpmap texture coordinates by detail coordinates
    o.TexBump.xy = v.vTex.xy * fDetailScaleLand * fBumpScale;   

    // Compute eye vector
    float3 vEye = normalize(vEyePoint - vWorldPos);

    // Compute normalized reflection vector
    // As we know the normal points straight up, we can simplify this
    // : float3 vReflection = normalize((2 * dot(vEye, vNormal) * vNormal) - vEye);
    float3 vReflection = normalize(float3(-vEye.x, (2 * vEye.y) - vEye.y, -vEye.z));

    // Tranform by environment matrix
    o.TexEnvMap.xy = mul(float4(vReflection.x, vReflection.y, vReflection.z, 1), mEnv);

    // Diffuse texture coordinates
    o.TexBase.xy = v.vTex;

    if (bLandDetail)
    {
        // Detail texture coordinates
        o.TexLandDetail.xy = v.vTex * fDetailScaleLand;
    }

    return o;
}

float4
TerrainPS(VS_OUTPUT Input,
          uniform bool bLand,
          uniform bool bWater,
          uniform bool bLandDetail,
          uniform bool bWaterDetail,
          uniform bool bReflect,
          uniform bool bBump) : COLOR
{
    float3 Color;
    float3 LandDetail;
    float3 WaterDetail;

    float4 ColorBase = tex2D(BaseSampler, Input.TexBase);

    if (bLandDetail)
    {
        LandDetail = tex2D(LandDetailSampler, Input.TexLandDetail);
    }

    if (bWaterDetail)
    {
        WaterDetail = tex2D(WaterDetailSampler, Input.TexWaterDetail);
    }

    if (bLand && bWater)
    {
        float fLand = ColorBase.a;
        if (bLandDetail && bWaterDetail)
        {
            Color = 2 * ColorBase * LandDetail  * fLand;
            Color += 2 * ColorBase * WaterDetail * (1 - fLand);
        }
        else if (bWaterDetail) // && !bLandDetail
        {
            Color = ColorBase * fLand;
            Color += 2 * ColorBase * WaterDetail * (1 - fLand);
        }
        else if (bLandDetail) // && !bWaterDetail
        {
            Color = 2 * ColorBase * LandDetail  * fLand;
            Color += ColorBase * (1 - fLand);
        }
        else // !bLandDetail && !bWaterDetail
        {
            Color = ColorBase;
        }

        if (bReflect)
        {
            Color += 0.5 * (tex2D(EnvMapSampler, Input.TexEnvMap) * (1 - fLand));
        }
    }
    else if (bWater)
    {
        // Water Only
        if (bWaterDetail)
        { 
            Color = (2 * ColorBase * WaterDetail);
        }
        else
        {
            Color = ColorBase;
        }

        if (bReflect)
        {
            Color += 0.5 * (tex2D(EnvMapSampler, Input.TexEnvMap));
        }
    }
    else
    {
        // Land Only
        if (bLandDetail)
        { 
            Color = (2 * ColorBase * LandDetail);
        }
        else
        {
            Color = ColorBase;
        }
    }

    return (float4(Color, 0));
}
