//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// Global shader.  Handles all cases
//

#include <GeneralShadow10.fxh>
#include <FuncLibrary.fxh>

// Declare inputs
struct VS_INPUT
{
#if VT_VERTEX
    float4  vPosition : POSITION;
    float4  vNormal   : NORMAL;
    float2  Tex       : TEXCOORD;
#elif VT_VERTEXCOL
    float4  vPosition : POSITION;
    float4  vNormal   : NORMAL;
    float4  cColor    : COLOR;
    float2  Tex       : TEXCOORD;
#elif VT_LVERTEX
    float4  vPosition : POSITION;
    float4  cColor    : COLOR0;
    float4  cSpecular : COLOR1;
    float2  Tex       : TEXCOORD;
#elif VT_TLVERTEX
    float4  vPosition : POSITION;
    float4  cColor    : COLOR0;
    float4  cSpecular : COLOR1;
    float2  Tex       : TEXCOORD;
#elif VT_VERTEX_XYZCUV
    float4  vPosition : POSITION;
    float4  cColor    : COLOR0;
    float2  Tex       : TEXCOORD;
#elif VT_VERTEX_XYZUV
    float4  vPosition : POSITION;
    float2  Tex       : TEXCOORD;
#elif VT_VERTEX_XYZC
    float4  vPosition : POSITION;
    float4  cColor    : COLOR0;
#elif VT_VERTEX_BUMP
    float4  vPosition : POSITION;
    float4  vNormal   : NORMAL;
    float2  Tex      : TEXCOORD;
    float3  vBinormal : BINORMAL;
    float3  vTangent  : TANGENT;
#elif VT_VERTEX_SKIN
    float4  vPosition : POSITION;
    float4  vNormal   : NORMAL;
    float2  Tex       : TEXCOORD;
    float4  vBlendWeight : BLENDWEIGHT;
    uint4   vBlendIndices : BLENDINDICES;
#elif VT_VERTEX_BUMPSKIN
    float4  vPosition : POSITION;
    float4  vNormal   : NORMAL;
    float2  Tex      : TEXCOORD;
    float3  vBinormal : BINORMAL;
    float3  vTangent  : TANGENT;
    float4  vBlendWeight : BLENDWEIGHT;
    uint4   vBlendIndices : BLENDINDICES;
#elif VT_VERTEX_MATRIX
    float4  vPosition : POSITION;
    float4  vNormal   : NORMAL;
    float2  Tex       : TEXCOORD;
    float4x4 mTransform : mTransform;
#else
    error
#endif        
};

// Declare outputs
struct VS_OUTPUT
{
    float4 vPos     : SV_POSITION;
    float  fFog     : FOG;
};

struct VS_SM_OUTPUT
{
    float4 vPos     : SV_POSITION;
    float  fDepth   : TEXCOORD0;
};

struct PS_OUTPUT
{
    float4 RGBColor : SV_Target;  // Pixel color
};

#if defined(VT_VERTEX_SKIN) || defined(VT_VERTEX_BUMPSKIN)
#define VS_INPUT_CONTAINS_SKINNING
#endif 

// Vertex shader for flat (old-style) shadows
VS_OUTPUT FLAT_VS(const VS_INPUT In)
{
    VS_OUTPUT Out = (VS_OUTPUT) 0;

    float4 vPosition = In.vPosition;

    // Apply Skinning to the position if present.
    #if defined(VS_INPUT_CONTAINS_SKINNING)
        vPosition = SkinPosition(vPosition, 
                                 In.vBlendWeight, 
                                 g_rgmBones[In.vBlendIndices.x], 
                                 g_rgmBones[In.vBlendIndices.y], 
                                 g_rgmBones[In.vBlendIndices.z], 
                                 g_rgmBones[In.vBlendIndices.w]);
    #endif

    #if VT_VERTEX_MATRIX
    vPosition = mul(vPosition, In.mTransform);
    #endif

    #if VT_TLVERTEX
        Out.vPos = mul(float4(vPosition.xy, 0.0f, 1.0f), g_mDeviceToViewport);
    #else
        Out.vPos = mul(vPosition, g_mWorldViewProjection);
    #endif

    // Compute fog
    Out.fFog = FogVS(In.vPosition, g_mWorld, g_vCameraEyePoint);

    return Out;
}

// Vertex shader for volume shadow shell rendering
VS_OUTPUT ShadowFSVSPass(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    o.vPos = mul(float4(v.vPosition.xy, 0.0f, 1.0f), g_mDeviceToViewport);
    o.fFog = 1.0;

    return o;
}

// Vertex shader for volume shadow shell rendering
VS_OUTPUT VOL_VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPosition, g_mWorldViewProjection);

    // Push the Z back a bit to reduce fighting
    o.vPos.z = o.vPos.z + 0.001f; 

    // No fog for volume shadow
    o.fFog = 1.0;

    return o;
}

// Pixel shader for flat (old-style) shadows
PS_OUTPUT FLAT_PS(VS_OUTPUT Input)
{
    PS_OUTPUT o;
    o.RGBColor = float4(FogPS(float4(0, 0, 0, g_fShadowIntensity), Input.fFog, g_fFogDensity, g_cFog), g_fShadowIntensity);
    return o;
}

// Pixel shader for volume shadow shell rendering (front side)
PS_OUTPUT VOL_PS(void)
{
    PS_OUTPUT o;
    o.RGBColor = float4(0,0,0,0.5);
    return o;
}

// Pixel shader for volume shadow shell rendering (back side)
PS_OUTPUT VOL_PS2(void)
{
    PS_OUTPUT o;
    o.RGBColor = float4(1,1,1,0.5);
    return o;
}

// Pixel shader for volume shadow full screeen pass
PS_OUTPUT ShadowFSPass()
{
    PS_OUTPUT o;
    o.RGBColor = (float4(0, 0, 0, g_fShadowIntensity));
    return o;
}

PS_OUTPUT ShadowMapClearPS()
{
    PS_OUTPUT o;
    o.RGBColor = float4(99999.9f,99999.9f,99999.9f,1.0f);
    return o;
}

// Basic Common State Blocks
RasterizerState rsCullCCW   { CullMode = Front; };
RasterizerState rsCullCW    { CullMode = Back; };
RasterizerState rsCullNONE  { CullMode = None; };

BlendState SrcAlphaBlend
{
        BlendEnable[0] = TRUE;
        SrcBlend = SRC_ALPHA;
        DestBlend = INV_SRC_ALPHA;
        BlendOp = ADD;
        SrcBlendAlpha = ZERO;
        DestBlendAlpha = ZERO;
        BlendOpAlpha = ADD;
        RenderTargetWriteMask[0] = 0x07; /*RED | GREEN | BLUE*/
};

BlendState NoAlphaBlend
{
        BlendEnable[0] = FALSE;
        SrcBlend = ONE;
        DestBlend = ZERO;
        BlendOp = ADD;
        SrcBlendAlpha = ZERO;
        DestBlendAlpha = ZERO;
        BlendOpAlpha = ADD;
        RenderTargetWriteMask[0] = 0x07; /*RED | GREEN | BLUE*/
};

BlendState NoOutput
{
        BlendEnable[0] = FALSE;
        SrcBlend = SRC_ALPHA;
        DestBlend = INV_SRC_ALPHA;
        BlendOp = ADD;
        SrcBlendAlpha = ZERO;
        DestBlendAlpha = ZERO;
        BlendOpAlpha = ADD;
        RenderTargetWriteMask[0] = 0x00; 
};


// Flat Shadowing Techniques
#if defined(SHD_SHADOW_FLAT)

    DepthStencilState dsFlatShadowStencil
    {
        DepthEnable = true;
        DepthWriteMask = ZERO;
        DepthFunc = LESS_EQUAL;
    
        StencilEnable = true;
        StencilReadMask = 0xFF;
        StencilWriteMask = 0xFF;
    
        FrontFaceStencilFunc = Equal;
        FrontFaceStencilPass = Incr;
        FrontFaceStencilFail = Keep;
        FrontFaceStencilDepthFail = Keep;
    
        BackFaceStencilFunc = Equal;
        BackFaceStencilPass = Incr;
        BackFaceStencilFail = Keep;
        BackFaceStencilDepthFail = Keep;
    };

    technique10 FlatShadowStencil
    {
        pass P0
        {
            SetVertexShader( CompileShader( vs_4_0, FLAT_VS( ) ) );
            SetGeometryShader( NULL );
            SetPixelShader( CompileShader( ps_4_0, FLAT_PS( ) ) );
            
            SetRasterizerState( rsCullNONE );
            SetBlendState( SrcAlphaBlend, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xFFFFFFFF );
            SetDepthStencilState(dsFlatShadowStencil, 0x00);
        }
    }

#endif // SHD_SHADOW_FLAT

#if defined(SHD_SHADOW_VOL_FS)

    DepthStencilState dsVolShadowFSStencil
    {
        DepthEnable = false;
        DepthWriteMask = ZERO;
        DepthFunc = Less;

        StencilEnable = true;
        StencilReadMask = 0xFF;
        StencilWriteMask = 0xFF;

        FrontFaceStencilFunc = Not_Equal;
        FrontFaceStencilPass = Keep;
        FrontFaceStencilFail = Keep;
        FrontFaceStencilDepthFail = Keep;

        BackFaceStencilFunc = Not_Equal;
        BackFaceStencilPass = Keep;
        BackFaceStencilFail = Keep;
        BackFaceStencilDepthFail = Keep;
    };
        
    technique10 VolShadowFS
    {
        pass P0
        {
                SetVertexShader( CompileShader( vs_4_0, ShadowFSVSPass( ) ) );
                SetGeometryShader( NULL );
                SetPixelShader( CompileShader( ps_4_0, ShadowFSPass( ) ) );
                
                SetRasterizerState( rsCullNONE );
                SetBlendState( SrcAlphaBlend, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xFFFFFFFF );
                SetDepthStencilState(dsVolShadowFSStencil, 0x00);
        }
    }

#endif // SHD_SHADOW_VOL_FS

#if defined(SHD_SHADOW_VOL)

    DepthStencilState dsVolShadowStencil
    {
        DepthEnable = true;
        DepthWriteMask = ZERO;
        DepthFunc = Less;

        StencilEnable = true;
        StencilReadMask = 0xFF;
        StencilWriteMask = 0xFF;

        FrontFaceStencilFunc = Always;
        FrontFaceStencilPass = Keep;
        FrontFaceStencilFail = Keep;
        FrontFaceStencilDepthFail = Incr;

        BackFaceStencilFunc = Always;
        BackFaceStencilPass = Keep;
        BackFaceStencilFail = Keep;
        BackFaceStencilDepthFail = Decr;
    };

    technique10 VolShadow
    {
        // CCW Back Pass
        pass P0
        {
            SetVertexShader( CompileShader( vs_4_0, VOL_VS( ) ) );
            SetGeometryShader( NULL );
            SetPixelShader( CompileShader( ps_4_0, VOL_PS( ) ) );
            
            SetRasterizerState( rsCullNONE );
            SetBlendState( NoOutput, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xFFFFFFFF );
            SetDepthStencilState(dsVolShadowStencil, 0x00);
        }
    }

#endif // SHD_SHADOW_VOL

#if defined(SHD_SHADOW_SM)

    struct SM_PS_OUTPUT
    {
        float fDepth : SV_TARGET;
    };

    VS_SM_OUTPUT ShadowMapVS(const VS_INPUT In)
    {
        VS_SM_OUTPUT Out = (VS_SM_OUTPUT) 0;

        float4 vPosition = In.vPosition;

        // Apply Skinning to the position if present.
        #if defined(VS_INPUT_CONTAINS_SKINNING)
            vPosition = SkinPosition(vPosition, 
                                     In.vBlendWeight, 
                                     g_rgmBones[In.vBlendIndices.x], 
                                     g_rgmBones[In.vBlendIndices.y], 
                                     g_rgmBones[In.vBlendIndices.z], 
                                     g_rgmBones[In.vBlendIndices.w]);
        #endif

        #if VT_VERTEX_MATRIX
            vPosition = mul(vPosition, In.mTransform);
        #endif

        #if VT_TLVERTEX
            Out.vPos = mul(float4(vPosition.xy, 0.0f, 1.0f), g_mDeviceToViewport);
        #else
            Out.vPos = mul(vPosition, g_mWorldViewProjection);
        #endif

        // Use linear distance instead of projected distance
        // We have better precision if we stay linear rather than work in "z-buffer space".
        Out.fDepth = mul(vPosition, g_mWorldView).z / g_fShadowRange;

        return Out;
    }

    SM_PS_OUTPUT ShadowMapPS(const VS_SM_OUTPUT In)
    {
        SM_PS_OUTPUT Out = (SM_PS_OUTPUT) 0;
        Out.fDepth = In.fDepth;
        return Out;
    }

    technique10 ShadowMap
    {
        pass P0
        {
            SetBlendState( NoAlphaBlend, float4( 0.0f, 0.0f, 0.0f, 0.0f ), 0xFFFFFFFF );
            SetVertexShader(CompileShader(vs_4_0, ShadowMapVS()));
            SetGeometryShader(NULL);
            SetPixelShader(CompileShader(ps_4_0, ShadowMapPS()));
        }
    }

#endif // SHD_SHADOW_SM

#if defined(SHD_SHADOW_SM_CLEAR)

    struct SM_CLEAR_VS_OUTPUT
    {
        float4 vPosition : SV_POSITION;
    };
    
    struct SM_CLEAR_PS_OUTPUT
    {
        float fDepth : SV_TARGET;
    };

    SM_CLEAR_VS_OUTPUT ShadowMapClearVS(const VS_INPUT v)
    {
        SM_CLEAR_VS_OUTPUT Out = (SM_CLEAR_VS_OUTPUT) 0;

        #if VT_TLVERTEX
            // We want the Z coordinate to be 1 - i.e. at the far clip plane.
            Out.vPosition = mul(float4(v.vPosition.xy, 1.0f, 1.0f), g_mDeviceToViewport);
        #else
            // The shadow map clear shader should only
            // ever see pretransformed vertices.
            INVALID_SHADER;
        #endif

        return Out;
    }
    
    SM_CLEAR_PS_OUTPUT ShadowMapClearPS(const SM_CLEAR_VS_OUTPUT In)
    {
        SM_CLEAR_PS_OUTPUT Out = (SM_CLEAR_PS_OUTPUT) 0;
        Out.fDepth = 1;
        return Out;
    }

    technique10 ShadowMapClear
    {
        pass P0
        {
            SetRasterizerState(rsCullNONE);
            SetVertexShader(CompileShader(vs_4_0, ShadowMapClearVS()));
            SetGeometryShader(NULL);
            SetPixelShader(CompileShader(ps_4_0, ShadowMapClearPS()));
        }
    }

#endif // SHD_SHADOW_SM_CLEAR
