//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// Global shader.  Handles all cases
//

#include <Common.fxh>
#include <MaterialDecl.fxh>
#include <D3D9Texture.fxh>
#include <FuncLibrary.fxh>

string XFile = "sjlcube.x";

const bool EffectDeclaration
<
    string Name      = "GeneralShadow";
    string Class     = "Basic";
    bool   NewMaterialUsage = true;
> = true;

// Declare inputs
struct VS_INPUT
{
    float4 vPos       : POSITION;
    #if defined(SHD_SHADOW_SKIN)
        float4 vBlendWeight   : BLENDWEIGHT;
        float4 vBlendIndices  : BLENDINDICES;
    #endif
};

// Declare outputs
struct VS_OUTPUT
{
    float4 vPos     : POSITION;
    float  fFog     : FOG;
};
struct VS_SM_OUTPUT
{
    float4 vPos     : POSITION;
    float  fDepth   : TEXCOORD0;
};

// Vertex shader for flat (old-style) shadows
VS_OUTPUT FLAT_VS(const VS_INPUT v)
{
    float4 vPos = v.vPos;
    VS_OUTPUT o = (VS_OUTPUT)0;

    // Apply skinning on position (if needed
    #if defined(SHD_SHADOW_SKIN)
    // Convert D3DCOLOR to UBYTE4. We can't use UBYTE4 types directly because
    // most 1.x shader hardware does not support it. As a sollution, we use
    // D3DCOLOR and use the D3DCOLORtoUBYTE4 macro to tweak the values back.
    float4 vBlendWeight = D3DCOLORtoUBYTE4(v.vBlendWeight) / 255.0f;
    float4 vBlendIndices = D3DCOLORtoUBYTE4(v.vBlendIndices);

    vPos = SkinPosition(vPos, 
                        vBlendWeight, 
                        mBones[vBlendIndices.x], 
                        mBones[vBlendIndices.y], 
                        mBones[vBlendIndices.z], 
                        mBones[vBlendIndices.w]);
    #endif

    // Transform to clip space
    o.vPos = mul(vPos, mFullProj);

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w, FOG_END, FOG_RECIP, FOG_SELECT_VERTEX, FOG_SELECT_TABLE);

    return o;
}

// Vertex shader for volume shadow shell rendering
VS_OUTPUT VOL_VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    // Transform to clip space
    o.vPos = mul(v.vPos, mFullProj);

    // Push the Z back a bit to reduce fighting
    o.vPos.z = o.vPos.z + 0.001f; 

    // No fog for volume shadow
    o.fFog = 1.0;

    return o;
}

VS_SM_OUTPUT SM_VS(const VS_INPUT v)
{
    VS_SM_OUTPUT o = (VS_SM_OUTPUT)0;

    float4 vPos;

    #if defined(SHD_SHADOW_SKIN)
    {
        // Convert D3DCOLOR to UBYTE4. We can't use UBYTE4 types directly because
        // most 1.x shader hardware does not support it. As a sollution, we use
        // D3DCOLOR and use the D3DCOLORtoUBYTE4 macro to tweak the values back.
        float4 vBlendWeight = D3DCOLORtoUBYTE4(v.vBlendWeight) / 255.0f;
        float4 vBlendIndices = D3DCOLORtoUBYTE4(v.vBlendIndices);

        // Apply skinning on position
        vPos = SkinPosition(v.vPos, 
                            vBlendWeight, 
                            mBones[vBlendIndices.x], 
                            mBones[vBlendIndices.y], 
                            mBones[vBlendIndices.z], 
                            mBones[vBlendIndices.w]);
    }
    #else
    {
        vPos = v.vPos;
    }
    #endif

    // Transform to clip space
    o.vPos = mul(vPos, mFullProj);

    // Use linear distance instead of projected distance
    // We have better precision if we stay linear rather than work in "z-buffer space".
    o.fDepth = mul(vPos, mWorldView).z / fShadowRange;

    return o;
}

float4 SM_PS(float fDepth:TEXCOORD0) : COLOR
{
    return float4(fDepth, 0, 0, 1);
}

// Pixel shader for flat (old-style) shadows
float4 FLAT_PS(void) : COLOR
{
    return (float4(0, 0, 0, fShadowIntensity));
}

// Pixel shader for volume shadow shell rendering (front side)
float4 VOL_PS(void) : COLOR
{
    return float4(0,0,0,0.5);
}

// Pixel shader for volume shadow shell rendering (back side)
float4 VOL_PS2(void) : COLOR
{
    return float4(1,1,1,0.5);
}

// Pixel shader for volume shadow full screeen pass
float4 ShadowFSPass() : COLOR
{
    return (float4(0, 0, 0, fShadowIntensity));
}

float4 ShadowMapClearPS() : COLOR
{
    return float4(99999.9f,99999.9f,99999.9f,1.0f);
}


// Flat Shadowing Techniques
#if defined(SHD_SHADOW_FLAT)
    technique FlatShadowStencil
    <
    int    Performance = EffectPerfLevel_MaxShader11;
    >
    {
        pass P0
        {
            #include <OneStageDefault.fxh>

            VertexShader = compile vs_1_1 FLAT_VS();
            PixelShader  = compile ps_1_1 FLAT_PS();

            ResultArg[0]             = Current;
            AddressU[0]              = Wrap;
            AddressV[0]              = Wrap;
            MinFilter[0]             = Linear;
            MagFilter[0]             = Linear;
            MipFilter[0]             = Linear;
            MipMapLodBias[0]         = 0;
            BorderColor[0]           = 0;
            TexCoordIndex[0]         = 0;
            TextureTransformFlags[0] = 0;

            TextureFactor            = 0x80000000;
            ColorArg1[0]             = TFactor;
            ColorArg2[0]             = TFactor;
            ColorOp[0]               = SelectArg1;
            AlphaArg1[0]             = TFactor;
            AlphaArg2[0]             = TFactor;
            AlphaOp[0]               = SelectArg1;

            AlphaBlendEnable         = True;
            SrcBlend	             = SrcAlpha;
            DestBlend	             = InvSrcAlpha;

            NormalizeNormals         = True;
            ZWriteEnable             = False;
            ZEnable                  = True;
            ZFunc                    = LessEqual;
            SeparateAlphaBlendEnable = False;
            AlphaTestEnable          = False;
            AlphaFunc                = Always;
            AlphaRef                 = 0;
            SpecularEnable           = False;
            ColorWriteEnable         = Red | Green | Blue;
            //FogEnable                = True;
            CullMode                 = None;

            StencilEnable            = true;
            StencilFunc              = NotEqual;
            StencilRef               = 0x01;
            StencilMask              = 0xffffffff;
            StencilWriteMask         = 0xffffffff;
            StencilZFail             = Keep;
            StencilFail              = Keep;
            StencilPass              = Replace;
        }
    }

    technique FlatShadowTFactor
    <
    >
    {
        pass P0
        {
            #include <OneStageDefault.fxh>

            ResultArg[0]             = Current;
            AddressU[0]              = Wrap;
            AddressV[0]              = Wrap;
            MinFilter[0]             = Linear;
            MagFilter[0]             = Linear;
            MipFilter[0]             = Linear;
            MipMapLodBias[0]         = 0;
            BorderColor[0]           = 0;
            TexCoordIndex[0]         = 0;
            TextureTransformFlags[0] = 0;

            TextureFactor            = 0x80000000;
            ColorArg1[0]             = TFactor;
            ColorArg2[0]             = TFactor;
            ColorOp[0]               = SelectArg1;
            AlphaArg1[0]             = TFactor;
            AlphaArg2[0]             = TFactor;
            AlphaOp[0]               = SelectArg1;

            AlphaBlendEnable         = True;
            SrcBlend	             = SrcAlpha;
            DestBlend	             = InvSrcAlpha;

            NormalizeNormals         = True;
            ZWriteEnable             = False;
            ZEnable                  = True;
            ZFunc                    = LessEqual;
            SeparateAlphaBlendEnable = False;
            AlphaTestEnable          = False;
            AlphaFunc                = Always;
            AlphaRef                 = 0;
            SpecularEnable           = False;
            ColorWriteEnable         = Red | Green | Blue;
            FogEnable                = True;
            CullMode                 = None;

            StencilEnable            = False;
        }
    }

    technique FlatShadowFallback
    <
    >
    {
        pass P0
        {
            #include <OneStageDefault.fxh>

            ResultArg[0]             = Current;
            AddressU[0]              = Wrap;
            AddressV[0]              = Wrap;
            MinFilter[0]             = Linear;
            MagFilter[0]             = Linear;
            MipFilter[0]             = Linear;
            MipMapLodBias[0]         = 0;
            BorderColor[0]           = 0;
            TexCoordIndex[0]         = 0;
            TextureTransformFlags[0] = 0;

            ColorOp[0]               = Disable;
            AlphaArg2[0]             = Diffuse;
            AlphaOp[0]               = SelectArg2;

            AlphaBlendEnable         = True;
            SrcBlend	             = SrcAlpha;
            DestBlend	             = InvSrcAlpha;

            NormalizeNormals         = True;
            ZWriteEnable             = False;
            ZEnable                  = True;
            ZFunc                    = LessEqual;
            SeparateAlphaBlendEnable = False;
            AlphaTestEnable          = False;
            AlphaFunc                = Always;
            AlphaRef                 = 0;
            SpecularEnable           = False;
            ColorWriteEnable         = Red | Green | Blue;
            FogEnable                = True;
            CullMode                 = None;

            StencilEnable            = False;
        }
    }
#endif

#if defined(SHD_SHADOW_VOL_FS)
technique VolShadowFS
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_1_1 ShadowFSPass();

        FogEnable               = False;
        ZFunc                   = Less;
        ZWriteEnable            = False;
        ZEnable                 = False;

        AlphaTestEnable         = True;
        AlphaBlendEnable        = True;
        SrcBlend                = SrcAlpha;
        DestBlend               = InvSrcAlpha;
        BlendOp                 = Add;

        CullMode                = None;
        ColorWriteEnable        = Red|Green|Blue|Alpha;

        StencilEnable           = True;
        StencilFunc             = NotEqual;
        StencilRef              = 0;
        StencilMask             = 0xFFFFFFFF; 
        StencilWriteMask        = 0xFFFFFFFF;
        StencilFail             = Keep;
        StencilPass             = Keep;
        StencilZFail            = Keep;
    }
}
#endif



#if defined(SHD_SHADOW_VOL)
// Volume shadow to stencil render
technique VolShadow
<
    int    Performance = EffectPerfLevel_MaxShader11;
>
{
        // CCW Back Pass
    pass P0
    {
        VertexShader = compile vs_1_1 VOL_VS();
        PixelShader  = compile ps_1_1 VOL_PS();

        StencilEnable    = True;
        StencilFunc      = Always;
        StencilFail      = Keep;
        StencilPass      = Keep;
        StencilZFail     = Incr;
        StencilRef       = 0;
        StencilMask      = 0xFFFFFFFF; // Keep the last bit for other tasks such as old shadows
        StencilWriteMask = 0xFFFFFFFF;
        CullMode         = Ccw;//Cw;
        AlphaTestEnable  = False;
        AlphaBlendEnable = False;
        ColorWriteEnable = 0;//Red|Green|Blue|Alpha;
        ZFunc            = Less;
        ZWriteEnable     = False;
    }

        // CW Front Pass
    pass P1
    {
        VertexShader = compile vs_1_1 VOL_VS();
        PixelShader  = compile ps_1_1 VOL_PS2();

        StencilEnable    = True;
        StencilFunc      = Always;
        StencilFail      = Keep;
        StencilPass      = Keep;
        StencilZFail     = Decr;
        StencilRef       = 0;
        StencilMask      = 0xFFFFFFFF; // Keep the last bit for other tasks such as old shadows
        StencilWriteMask = 0xFFFFFFFF;
        CullMode         = Cw;//Ccw;
        AlphaTestEnable  = False;
        AlphaBlendEnable = False;
        ColorWriteEnable = 0;//Red|Green|Blue|Alpha;
        ZFunc            = Less;
        ZWriteEnable     = False;
    }
}
#endif

#if defined(SHD_SHADOW_SM)
technique ShadowMap
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        #include <OneStageDefault.fxh>

        VertexShader = compile vs_2_0 SM_VS();
        PixelShader  = compile ps_2_0 SM_PS();

        AlphaBlendEnable            = False;
        SrcBlend                    = SrcAlpha;
        DestBlend                   = InvSrcAlpha;
        ZEnable                     = True;
        ZWriteEnable                = True;
        ZFunc                       = (State_ZFunc);

        ResultArg[0]                = Current;
        AddressU[0]                 = Wrap;
        AddressV[0]                 = Wrap;
        MinFilter[0]                = Linear;
        MagFilter[0]                = Linear;
        MipFilter[0]                = Linear;
        MipMapLodBias[0]            = 0;
        BorderColor[0]              = 0;
        TexCoordIndex[0]            = 0;
        TextureTransformFlags[0]    = 0;

        TextureFactor               = 0x80000000;
        ColorArg1[0]                = TFactor;
        ColorArg2[0]                = TFactor;
        ColorOp[0]                  = SelectArg1;
        AlphaArg1[0]                = TFactor;
        AlphaArg2[0]                = TFactor;
        AlphaOp[0]                  = SelectArg1;

        AlphaTestEnable         = False;
        AlphaFunc               = Always;
        AlphaRef                = 0;
        SpecularEnable          = False;
        ColorWriteEnable        = Red|Green|Blue|Alpha;

        // Use standard cull mode so VC can idsable shadow casting for elements such as glass windows...
        CullMode                    = (State_CullMode);
    }
}
#endif

#if defined(SHD_SHADOW_SM_CLEAR)
technique ShadowMapClear
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        PixelShader  = compile ps_1_1 ShadowMapClearPS();

        FogEnable               = False;
        StencilEnable           = False;
        ZFunc                   = Less;
        ZWriteEnable            = False;
        ZEnable                 = False;

        AlphaTestEnable         = False;
        AlphaBlendEnable        = False;
        SrcBlend                = SrcAlpha;
        DestBlend               = InvSrcAlpha;
        CullMode                = None;
        ColorWriteEnable        = Red|Green|Blue|Alpha;
    }
}
#endif
