//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// Global shader.  Handles all cases
//

#include <Common.fxh>
#include <MaterialDecl.fxh>
#include <D3D9Texture.fxh>
#include <FuncLibrary.fxh>

string XFile = "sjlcube.x";

#if defined(SHD_FRESNEL_DIFFUSE) || defined(SHD_FRESNEL_SPECULAR) || defined(SHD_FRESNEL_ENV)
#define PS_USES_FRESNEL true
#endif

#if defined(PS_USES_FRESNEL) || defined(SHD_ENV) || defined(SHD_SPECULAR)
#define PS_NEEDS_EYE_DIR true
#endif

#if !defined(SHD_BUMP) && !defined(SHD_VERTICAL_NORMAL) && !defined(SHD_PRELIT)
#define PS_NEEDS_VERTEX_NORMAL true
#endif

#if defined(SHD_BUMP) && !defined(SHD_VERTICAL_NORMAL)
#define PS_NEEDS_TANSPACE
#endif

const bool EffectDeclaration
<
    string Name      = "General";
    string Class     = "Basic";
    string Features  = "Bump";
    string Fallback  = "Assert";
    bool   NewMaterialUsage = true;
> = true;

#define vLightDir vSunVectorWorld

// Declare inputs
struct VS_INPUT
{
    float4 vPos       : POSITION;
#if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormal    : NORMAL;
#endif

#if defined(SHD_PRELIT)
    float4 vDiffuse   : COLOR0;
#endif
    float2 vTex       : TEXCOORD0;

#ifdef PS_NEEDS_TANSPACE
    float3 vBinormal  : BINORMAL;
    float3 vTangent   : TANGENT
    ;
#endif

#if defined(SHD_SKIN)
    float4 vBlendWeight   : BLENDWEIGHT;
    float4 vBlendIndices  : BLENDINDICES;
#endif

};

// Declare outputs
struct PS_INPUT
{
    float2 TexBase  : TEXCOORD0;

#ifdef SHD_DETAIL
    float2 TexDetail  : TEXCOORD1;
#endif

#ifdef SHD_BUMP
    float2 TexBump  : TEXCOORD2;
#else
    float4 cDiffuse   : COLOR0;
#endif

#if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormalWS  : TEXCOORD3;
#endif

#ifdef PS_NEEDS_EYE_DIR
    float3 vPosWS     : TEXCOORD4;
    float3 vEyePosWS  : TEXCOORD5;
#endif

#ifdef PS_NEEDS_TANSPACE
    float3 vBinormalWS : TEXCOORD6;
    float3 vTangentWS  : TEXCOORD7;
#endif

};

// Declare outputs
struct VS_OUTPUT
{
    float4 vPos     : POSITION;
    float  fFog     : FOG;

    PS_INPUT ToPix;
};

// Code
VS_OUTPUT
VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;
    float4 vPos = v.vPos;

    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormal = v.vNormal;
    #endif

    #ifdef SHD_SKIN

    // Convert D3DCOLOR to UBYTE4. We can't use UBYTE4 types directly because
    // most 1.x shader hardware does not support it. As a sollution, we use
    // D3DCOLOR and use the D3DCOLORtoUBYTE4 macro to tweak the values back.
    float4 vBlendWeight = D3DCOLORtoUBYTE4(v.vBlendWeight) / 255.0f;
    float4 vBlendIndices = D3DCOLORtoUBYTE4(v.vBlendIndices);

    // Apply skinning on position
    vPos = SkinPosition(vPos, 
                        vBlendWeight, 
                        mBones[vBlendIndices.x], 
                        mBones[vBlendIndices.y], 
                        mBones[vBlendIndices.z], 
                        mBones[vBlendIndices.w]);

    // Apply skinning on normal (if needed)
    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    vNormal = SkinNormal(vNormal,
                         vBlendWeight,
                         mBones[vBlendIndices.x], 
                         mBones[vBlendIndices.y], 
                         mBones[vBlendIndices.z], 
                         mBones[vBlendIndices.w]);
    #endif //normal

    #endif //skin

    // Transform to clip space
    o.vPos = mul(vPos, mFullProj);

    // Base texture coordinates
    o.ToPix.TexBase.xy = v.vTex;

    #ifdef SHD_DETAIL
    {
        o.ToPix.TexDetail.xy = v.vTex * DetailScale;
    }
    #endif

    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    {
        o.ToPix.vNormalWS = TransformVectorToSpace(vNormal, mWorld);
    }
    #endif

    #ifdef SHD_BUMP
    {
        o.ToPix.TexBump.xy = v.vTex * BumpScale;
        #ifdef PS_NEEDS_TANSPACE
        {
                        #ifdef SHD_SKIN
                        vTangent = SkinNormal(v.vTangent,
                                              vBlendWeight,
                                              mBones[vBlendIndices.x], 
                                              mBones[vBlendIndices.y], 
                                              mBones[vBlendIndices.z], 
                                              mBones[vBlendIndices.w]);
                        #else
                        float3 vTangent  = v.vTangent;
                        #endif

                        // Transform tangent into worldspace and calculate binormal
                        o.ToPix.vTangentWS  = TransformVectorToSpace(vTangent,  mWorld);
                        o.ToPix.vBinormalWS = cross(vNormalWS, o.ToPix.vTangentWS);
        }
        #endif
    }
    #else // !SHD_BUMP
    {
        #ifdef SHD_PRELIT
        {
            o.ToPix.cDiffuse = v.vDiffuse;
        }
        #else
        {
            float fDot;
            #ifdef SHD_VERTICAL_NORMAL
            {
                fDot = max(vLightDir.y, 0);
            }
            #else
            {
                fDot = saturate(dot(TransformVectorToSpace(vNormal, mWorld), vLightDir));
            }
            #endif

            o.ToPix.cDiffuse = float4( vSunAmbient + (vSunDirectional * fDot), 1.0);

            #ifndef SHD_BASE
            {
                o.ToPix.cDiffuse *= MatDiffuse;
            }
            #endif // !SHD_BASE
        }
        #endif // !SHD_PRELIT
    }
    #endif // !SHD_BUMP

    #ifdef PS_NEEDS_EYE_DIR
    {
        o.ToPix.vPosWS    = mul(vPos, mWorld);
        o.ToPix.vEyePosWS = vEyePoint;
    }
    #endif

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w, FOG_END, FOG_RECIP, FOG_SELECT_VERTEX, FOG_SELECT_TABLE);

    return o;
}

float4
PS(PS_INPUT Input) : COLOR
{
    //
    // Base texture and detail
    //
    float4 cBase;
    #ifdef SHD_BASE
    {
        cBase = tex2D(BaseSampler, Input.TexBase);
    }
    #else
    {
        cBase = float4(1, 1, 1, 1);
    }
    #endif

    #ifdef SHD_DETAIL
    {
        float3 cDetail = tex2D(DetailSampler, Input.TexDetail);
        cBase *= float4(2 * cDetail, 1);
    }
    #endif

    //
    // Build all terms
    //
    float4 cColor, cDiffuse, cFresnel;

    // Get required normals
    float3 vNormalWS, vVtxNormalWS;
    vVtxNormalWS = vNormalWS = float3(0, 1, 0);

    #if defined(SHD_BUMP)
    {
        float3 vNormalSurface = SampleBump(BumpSampler, Input.TexBump);

        #if !defined(SHD_VERTICAL_NORMAL)
        {
            vVtxNormalWS = normalize(Input.vNormalWS);

            // Transform surface normal into world space
            vNormalWS = TransformVectorByColumnMatrix(vNormalSurface, normalize(Input.vBinormalWS), -normalize(Input.vTangentWS), vVtxNormalWS);
        }
        #else
        {
            vNormalWS = TransformVectorToSpace(float3(vNormalSurface.r, vNormalSurface.b, -vNormalSurface.g), mWorld);
        }
        #endif
    }
    #else  // !SHD_BUMP
    {
        #if !defined(SHD_VERTICAL_NORMAL) && !defined(SHD_PRELIT) && (defined(PS_USES_FRESNEL) || defined(SHD_SPECULAR) || defined(SHD_ENV))
        {
            float3 vNorm = normalize(Input.vNormalWS);

            #if defined(PS_USES_FRESNEL)
            {
                vVtxNormalWS = vNorm;
            }
            #endif

            #if defined(SHD_SPECULAR) || defined(SHD_ENV)
            {
                vNormalWS = vNorm;
            }
            #endif
        }
        #endif
    }
    #endif // !SHD_BUMP

    float3 vEyeDirWS;
    #if defined(PS_NEEDS_EYE_DIR)
    {
        vEyeDirWS = normalize(Input.vEyePosWS - Input.vPosWS);
    }
    #endif

    #if defined(PS_USES_FRESNEL)
    {
        //
        // Fresnel term
        //

        // We will not bump the fresnel term as it is cheaper
        // not to and avoids sparklies
        cFresnel = CalculateFresnel(FresnelSampler, vEyeDirWS, vVtxNormalWS);
    }
    #endif

    //
    // Diffuse term
    //
    #if defined(SHD_BUMP)
    {
        // Perform diffuse calculation
        float fDot = saturate(dot(vNormalWS, vLightDir));
        cDiffuse = cBase * float4( vSunAmbient + (vSunDirectional * fDot), 1);
    }
    #else
    {
        // Not bumpmapping, so diffuse comes from the vertex
        cDiffuse = cBase * Input.cDiffuse;
    }
    #endif

    #ifdef SHD_PRELIT
    {
        cDiffuse.rgb *= g_fEmissiveCoefficient;
    }
    #endif

    // Need to have the specular color here in case the alpha is needed to modulate the diffuse
    #if defined(SHD_SPECULARMAP) || defined(SHD_BLENDENVBYSPECULARMAPALPHA)
    float4 cSpecularMap = tex2D(SpecularSampler, Input.TexBase);
    #endif

    #if defined(SHD_BLENDDIFFUSEVBYBASEALPHA)
    {
      cDiffuse *= cBase.a;
    }
    #else
    {
        #if defined(SHD_BLENDDIFFUSEBYINVSPECULARMAPALPHA)
        {
            cDiffuse *= (1 - cSpecularMap.a);
        }
        #endif
    }
    #endif

    #if defined(SHD_FRESNEL_DIFFUSE)
    {
        cDiffuse *= (1 - cFresnel);
    }
    #endif

    #if defined(SHD_BLENDENVBYINVBASEALPHA)
    {
        cDiffuse.a = MatDiffuse.a;
    }
    #endif

    cColor = cDiffuse;

    #if defined(SHD_SPECULAR)
    {
        //
        // Specular term
        //
        // Specular:  S = N.H ^ Power
        float3 cSpecular;
        #if defined(SHD_SPECULARMAP)
        {
            cSpecular = CalculateSpecularFromMap(vEyeDirWS, vLightDir, vNormalWS, cSpecularMap, MatSpecMapPowerScale, vSunDirectional);
        }
        #else
        {
            cSpecular = CalculateSpecular(vEyeDirWS, vLightDir, vNormalWS, SpecularPower, vSunDirectional, SpecularColor);
        }
        #endif

        #if defined(SHD_FRESNEL_SPECULAR)
        {
            cSpecular *= cFresnel;
        }
        #endif

        #if !defined (SHD_NOSPECULARBLOOM)
        {
            float fPower;

            #if defined(SHD_SPECULARMAP)
            {
                fPower = cSpecularMap.a * MatSpecMapPowerScale;
            }
            #else
            {
                fPower = SpecularPower;
            }
            #endif

            float fBloom;
            
            #if !defined(SHD_VERTICAL_NORMAL)
            {
                fBloom = CalculateSpecularBloom(vEyeDirWS, vLightDir, vNormalWS, fPower, MatSpecBloomFloor);
            }
            #else
            {
                fBloom = CalculateSpecularBloom(vEyeDirWS, vLightDir, fPower, MatSpecBloomFloor);
            }
            #endif
            if (fBloom > 1)
            {
                #if defined(SHD_SPECULARMAP)
                {
                    cSpecular += fBloom * cSpecularMap * vSunDirectional;
                }
                #else
                {
                    cSpecular += fBloom * SpecularColor * vSunDirectional;
                }
                #endif
            }
        }
        #endif

        cColor += float4(cSpecular, 0);
    }
    #endif

    #if defined(SHD_ENV)
    {
        //
        // Environment term
        //
        float3 cReflection = CalculateEnv(EnvSampler, vEyeDirWS, vNormalWS, vSunDirectional);

        #if defined(SHD_BLENDENVBYINVBASEALPHA)
        {
            cReflection *= (1 - cBase.a);
        }
        #else
        {
            #if defined(SHD_BLENDENVBYSPECULARMAPALPHA)
            {
                cReflection *= cSpecularMap.a;
            }
            #endif
        }
        #endif

        #if defined(SHD_FRESNEL_ENV)
        {
            cReflection *= cFresnel;
        }
        #endif

        #if defined(SHD_PRECIP_BLEND)
        {
            #if defined(SHD_PRECIP_OFFSET)
            {
                cReflection = lerp(cReflection * MatPrecipOffset, cReflection, GlobalPrecipLevel);
            }
            #else
            {
                cReflection *= GlobalPrecipLevel;
            }
            #endif
        }
        #endif

        cColor += float4(cReflection, 0) * EnvLevelScale;
    }
    #endif

    #if defined( SHD_ADDITIVE_EMISSIVEMAP ) || defined( SHD_BLENDED_EMISSIVEMAP ) || defined( SHD_MULTIPLY_EMISSIVEMAP )
    {
        float4 cEmissive = tex2D(EmissiveSampler, Input.TexBase);

        #if defined( SHD_ADDITIVE_EMISSIVEMAP )
        {
            cColor.r += cEmissive.r * g_fEmissiveCoefficient;
            cColor.g += cEmissive.g * g_fEmissiveCoefficient;
            cColor.b += cEmissive.b * g_fEmissiveCoefficient;
        }
        #elif defined( SHD_BLENDED_EMISSIVEMAP )
        {
            cColor = lerp(cEmissive, cColor, State_BlendedEmissiveFactor);
        }
        #elif defined( SHD_MULTIPLY_EMISSIVEMAP )
        {
            float fDayNightRatio = 1 - State_BlendedEmissiveFactor;     // 1 is night, 0 is day
            float4 cEmissiveMapped = (cDiffuse + (fDayNightRatio * cEmissive)) * cBase;

            cColor = float4(cEmissiveMapped.r, cEmissiveMapped.g, cEmissiveMapped.b, cColor.a);
        }
        #endif

    }
    #endif

    #if defined(SHD_FINAL_ALPHA_BLEND)
    {
        cColor.a *= State_FinalAlphaBlendFactor;
    }
    #endif
    
    // Bloom
    #if defined(SHD_BLOOM_COPY) || defined(SHD_BLOOM_MODALPHA)
    {
        float fBloom = 1;

        #if defined(SHD_BLOOM_MODALPHA)
        {
            fBloom = cBase.a * g_fEmissiveCoefficient;
        }
        #endif

        cColor.rgb *= fBloom;
    }
    #endif

    //return (min(cColor, 65504));
    return cColor;
}

technique T0
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        #include <DefaultStates.fxh>

        FogEnable = false;
        VertexShader = compile vs_3_0 VS();
        PixelShader  = compile ps_3_0 PS();

        AlphaBlendEnable = (State_AlphaBlendEnable);
        SrcBlend         = (State_SrcBlend);
        DestBlend        = (State_DstBlend);
    }
}
