//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// Global Fallback shader.
//

#include <Common.fxh>
#include <MaterialDecl.fxh>
#include <D3D9Texture.fxh>
#include <FuncLibrary.fxh>

string XFile = "sjlcube.x";

const bool EffectDeclaration
<
    string Name      = "GeneralFallback";
    string Class     = "Basic";
    string Features  = "";
    string Fallback  = "Texture";
    bool   NewMaterialUsage = true;
> = true;

#define vLightDir vSunVectorWorld

// Declare inputs
struct VS_INPUT
{
    float4 vPos       : POSITION;
#if !defined(SHD_PRELIT)
    float3 vNormal    : NORMAL;
#else
    float4 vDiffuse   : COLOR0;
#endif

#if defined(SHD_SKIN)
    float4 vBlendWeight   : BLENDWEIGHT;
    float4 vBlendIndices  : BLENDINDICES;
#endif

#if defined(SHD_BASE) || defined(SHD_DETAIL) || defined(SHD_ADDITIVE_EMISSIVEMAP) || defined(SHD_BLENDED_EMISSIVEMAP) || defined(SHD_MULTIPLY_EMISSIVEMAP)
    float2 vTex       : TEXCOORD0;
#endif

#if defined(SHD_APPLY_Y_BIAS)
    float1 vYBias        : TEXCOORD2;
#endif

#if defined(SHD_APPLY_VERTEX_ALPHA_FACTOR)
    float1 vAlphaFactor  : TEXCOORD3;
#endif

};

struct PS_INPUT
{
    float4 cDiffuse   : COLOR0;
    float2 TexBase    : TEXCOORD0;
#ifdef SHD_DETAIL
    float2 TexDetail  : TEXCOORD1;
#endif
#if defined(SHD_ADDITIVE_EMISSIVEMAP) || defined(SHD_BLENDED_EMISSIVEMAP) || defined(SHD_MULTIPLY_EMISSIVEMAP)
    float2 TexEmissive : TEXCOORD2;
#endif
};

// Declare outputs
struct VS_OUTPUT
{
    float4 vPos     : POSITION;
    float  fFog     : FOG;

    PS_INPUT ToPix;
};

// Code
VS_OUTPUT
VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;

    // variable to accumulate skin influences
    float4  vPos = v.vPos;

    #if !defined(SHD_PRELIT) && !defined(SHD_VERTICAL_NORMAl)
    float3  vNormal = v.vNormal;
    #endif

    #ifdef SHD_SKIN
    {
        // Convert D3DCOLOR to UBYTE4. We can't use UBYTE4 types directly because
        // most 1.x shader hardware does not support it. As a sollution, we use
        // D3DCOLOR and use the D3DCOLORtoUBYTE4 macro to tweak the values back.
        float4 vBlendWeight = D3DCOLORtoUBYTE4(v.vBlendWeight) / 255.0f;
        float4 vBlendIndices = D3DCOLORtoUBYTE4(v.vBlendIndices);

        // Apply skinning on position
        vPos = SkinPosition(vPos,
                            vBlendWeight,
                            mBones[vBlendIndices.x],
                            mBones[vBlendIndices.y],
                            mBones[vBlendIndices.z],
                            mBones[vBlendIndices.w]);

        #if !defined(SHD_PRELIT) && !defined(SHD_VERTICAL_NORMAL)
        {
            // Apply skinning on normal (if needed)
            vNormal = SkinNormal(vNormal,
                                 vBlendWeight,
                                 mBones[vBlendIndices.x],
                                 mBones[vBlendIndices.y],
                                 mBones[vBlendIndices.z],
                                 mBones[vBlendIndices.w]);
        }
        #endif //normal
    }
    #endif //skin

    // Vertical bias from stream
    #ifdef SHD_APPLY_Y_BIAS
    {
        vPos.y += v.vYBias.x;
    }
    #endif


    // Transform to clip space
    o.vPos = mul(vPos, mFullProj);


    #ifdef SHD_BASE
    {
        o.ToPix.TexBase.xy = v.vTex;
    }

    #endif

    #ifdef SHD_DETAIL
    {
        o.ToPix.TexDetail.xy = v.vTex * DetailScale;
    }
    #endif

    #if defined(SHD_ADDITIVE_EMISSIVEMAP) || defined(SHD_BLENDED_EMISSIVEMAP) || defined(SHD_MULTIPLY_EMISSIVEMAP)
    {
        o.ToPix.TexEmissive = v.vTex;
    }
    #endif

    #ifdef SHD_PRELIT
    {
        o.ToPix.cDiffuse = v.vDiffuse;
    }
    #else
    {
        float fDot;
        #ifdef SHD_VERTICAL_NORMAL
        {
            fDot = max(vLightDir.y, 0);
        }
        #else
        {
            fDot = saturate(dot(TransformVectorToSpace(vNormal, mWorld), vLightDir));
        }
        #endif

        o.ToPix.cDiffuse = float4(vSunAmbient + (vSunDirectional * fDot), 1.0);

        #ifndef SHD_BASE
        {
            o.ToPix.cDiffuse *= MatDiffuse;
        }
        #endif // !SHD_BASE
    }
    #endif // !SHD_PRELIT

    // Alpha blend from stream
    #ifdef	SHD_APPLY_VERTEX_ALPHA_FACTOR
    {
	o.ToPix.cDiffuse.a *= v.vAlphaFactor.x;
    }
    #endif

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w, FOG_END, FOG_RECIP, FOG_SELECT_VERTEX, FOG_SELECT_TABLE);

    return o;
}

float4
PS(PS_INPUT Input) : COLOR
{
    //
    // Base texture and detail
    //
    float4 cBase;
    #ifdef SHD_BASE
    {
        cBase = tex2D(BaseSampler, Input.TexBase);
    }
    #else
    {
        cBase = float4(1, 1, 1, 1);
    }
    #endif

    #ifdef SHD_DETAIL
    {
        float3 cDetail = tex2D(DetailSampler, Input.TexDetail);
        cBase *= float4(2 * cDetail, 1);
    }
    #endif

    //
    // Build all terms
    //
    float4 cColor;

    //
    // Diffuse term
    //
    #if defined(SHD_BASE) || defined(SHD_DETAIL)
    {
        cColor = cBase * Input.cDiffuse;
    }
    #else
    {
        cColor = Input.cDiffuse;
    }
    #endif

    #if defined(SHD_BLENDENVBYINVBASEALPHA)
    {
        cColor.a = MatDiffuse.a;
    }
    #endif


    #if defined( SHD_ADDITIVE_EMISSIVEMAP ) || defined( SHD_BLENDED_EMISSIVEMAP ) || defined( SHD_MULTIPLY_EMISSIVEMAP )
    {
        float4 cEmissive = tex2D(EmissiveSampler, Input.TexEmissive);

        #if defined( SHD_ADDITIVE_EMISSIVEMAP )
        {
            cColor += float4(cEmissive.r, cEmissive.g, cEmissive.b, 0);
        }
        #elif defined( SHD_BLENDED_EMISSIVEMAP )
        {
            cColor = lerp(cEmissive, cColor, State_BlendedEmissiveFactor);
        }
        #elif defined( SHD_MULTIPLY_EMISSIVEMAP )
        {
            float4 cDiffuse = cBase * Input.cDiffuse;
            float fDayNightRatio = 1 - State_BlendedEmissiveFactor;     // 1 is night, 0 is day
            float4 cEmissiveMapped = (cDiffuse + (fDayNightRatio * cEmissive)) * cBase;

            cColor = float4(cEmissiveMapped.r, cEmissiveMapped.g, cEmissiveMapped.b, cColor.a);
        }
        #endif

    }
    #endif

    #if defined(SHD_FINAL_ALPHA_BLEND)
    {
        cColor.a *= State_FinalAlphaBlendFactor;
    }
    #endif

    return (cColor);
}

float4
PS_11(PS_INPUT Input) : COLOR
{
    //
    // Base texture
    //
    float4 cBase;
    #ifdef SHD_BASE
    {
        cBase = tex2D(BaseSampler, Input.TexBase);
    }
    #else
    {
        cBase = float4(1, 1, 1, 1);
    }
    #endif

    float4 cColor = cBase * Input.cDiffuse;

    #if defined( SHD_ADDITIVE_EMISSIVEMAP ) || defined( SHD_BLENDED_EMISSIVEMAP ) || defined( SHD_MULTIPLY_EMISSIVEMAP )
    {
        float4 cEmissive = tex2D(EmissiveSampler, Input.TexEmissive);

        #if defined( SHD_ADDITIVE_EMISSIVEMAP )
        {
            cColor += float4(cEmissive.r, cEmissive.g, cEmissive.b, 0);
        }
        #elif defined( SHD_BLENDED_EMISSIVEMAP )
        {
            cColor = lerp(cEmissive, cColor, State_BlendedEmissiveFactor);
        }
        #elif defined( SHD_MULTIPLY_EMISSIVEMAP )
        {
            float4 cDiffuse = cBase * Input.cDiffuse;
            float fDayNightRatio = 1 - State_BlendedEmissiveFactor;     // 1 is night, 0 is day
            float4 cEmissiveMapped = (cDiffuse + (fDayNightRatio * cEmissive)) * cBase;

            cColor = float4(cEmissiveMapped.r, cEmissiveMapped.g, cEmissiveMapped.b, cColor.a);
        }
        #endif
    }
    #else
        #if defined( SHD_FINAL_ALPHA_BLEND )
        {
            cColor.a *= State_FinalAlphaBlendFactor;
        }
        #endif
    #endif

    return (cColor);
}

technique T0
<
    int    Performance = EffectPerfLevel_MaxShader11;
>
{
    pass P0
    {
        #include <DefaultStates.fxh>

        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_1_4 PS();

        AlphaBlendEnable = (State_AlphaBlendEnable);
        SrcBlend         = (State_SrcBlend);
        DestBlend        = (State_DstBlend);
    }
}

technique T1
<
    int    Performance = EffectPerfLevel_MaxShader11;
>
{
    pass P0
    {
        #include <DefaultStates.fxh>

        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_1_1 PS_11();

        AlphaBlendEnable = (State_AlphaBlendEnable);
        SrcBlend         = (State_SrcBlend);
        DestBlend        = (State_DstBlend);
    }
}


