//------------------------------------------------------------------------------
// Copyright (c) Microsoft Corporation.  All Rights Reserved.
//------------------------------------------------------------------------------

#include <General10.fxh>
#include <D3D10Texture.fxh>
#include <FuncLibrary.fxh>
#include <FuncLibrary10.fxh>

/////////////////////////////////////////////////////////////
// VS_INPUT
struct VS_INPUT
{
#if VT_VERTEX
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD;
#elif VT_VERTEXCOL
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float4  cColor        : COLOR;
    float2  Tex           : TEXCOORD;
#elif VT_LVERTEX
    float4  vPosition     : POSITION;
    float4  cColor        : COLOR0;
    float4  cSpecular     : COLOR1;
    float2  Tex           : TEXCOORD;
#elif VT_TLVERTEX
    float4  vPosition     : POSITION;
    float4  cColor        : COLOR0;
    float4  cSpecular     : COLOR1;
    float2  Tex           : TEXCOORD;
#elif VT_VERTEX_XYZCUV
    float4  vPosition     : POSITION;
    float4  cColor        : COLOR0;
    float2  Tex           : TEXCOORD;
#elif VT_VERTEX_XYZUV
    float4  vPosition     : POSITION;
    float2  Tex           : TEXCOORD;
#elif VT_VERTEX_XYZC
    float4  vPosition     : POSITION;
    float4  cColor        : COLOR0;
#elif VT_VERTEX_BUMP
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD;
    float3  vBinormal     : BINORMAL;
    float3  vTangent      : TANGENT;
#elif VT_VERTEX_SKIN
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD;
    float4  vBlendWeight  : BLENDWEIGHT;
    uint4   vBlendIndices : BLENDINDICES;
#elif VT_VERTEX_BUMPSKIN
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD;
    float3  vBinormal     : BINORMAL;
    float3  vTangent      : TANGENT;
    float4  vBlendWeight  : BLENDWEIGHT;
    uint4   vBlendIndices : BLENDINDICES;
#elif VT_VERTEX_MATRIX
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD;
    float4x4 mTransform   : mTransform;
#elif VT_VERTEX_BUMP_Y_BIAS_ALPHA
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD0;
    float3  vBinormal     : BINORMAL;
    float3  vTangent      : TANGENT;
    float1  vYBias        : TEXCOORD1;
    float1  vAlphaFactor  : TEXCOORD2;
#elif VT_VERTEX_BUMP_Y_BIAS
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD0;
    float3  vBinormal     : BINORMAL;
    float3  vTangent      : TANGENT;
    float1  vYBias        : TEXCOORD1;
#elif VT_VERTEX_BUMP_ALPHA
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD0;
    float3  vBinormal     : BINORMAL;
    float3  vTangent      : TANGENT;
    float1  vAlphaFactor  : TEXCOORD1;
#elif VT_VERTEX_Y_BIAS_ALPHA
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD0;
    float1  vYBias        : TEXCOORD1;
    float1  vAlphaFactor  : TEXCOORD2;
#elif VT_VERTEX_Y_BIAS
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD0;
    float1  vYBias        : TEXCOORD1;
#elif VT_VERTEX_ALPHA
    float4  vPosition     : POSITION;
    float4  vNormal       : NORMAL;
    float2  Tex           : TEXCOORD0;
    float1  vAlphaFactor  : TEXCOORD1;
#else
    error
#endif
};

#if defined(VT_VERTEX) || defined(VT_VERTEXCOL) || defined(VT_VERTEX_BUMP) || defined(VT_VERTEX_SKIN) || defined(VT_VERTEX_BUMPSKIN) || defined(VT_VERTEX_MATRIX) || defined(VT_VERTEX_Y_BIAS) || defined(VT_VERTEX_BUMP_Y_BIAS_ALPHA) || defined(VT_VERTEX_BUMP_Y_BIAS) || defined(VT_VERTEX_BUMP_ALPHA)
#define VS_INPUT_CONTAINS_NORMAL
#endif

#if defined(VT_VERTEX_SKIN) || defined(VT_VERTEX_BUMPSKIN)
#define VS_INPUT_CONTAINS_SKINNING
#endif

#if defined(VT_VERTEX) || defined(VT_LVERTEX) || defined(VT_TLVERTEX) || defined(VT_VERTEX_XYZCUV) || defined(VT_VERTEX_XYZUV) || defined(VT_VERTEX_BUMP) || defined(VT_VERTEX_SKIN) || defined(VT_VERTEX_BUMPSKIN) || defined(VT_VERTEX_MATRIX) || defined(VT_VERTEX_BUMP_Y_BIAS_ALPHA) || defined(VT_VERTEX_BUMP_Y_BIAS) || defined(VT_VERTEX_BUMP_ALPHA) || defined(VT_VERTEX_Y_BIAS_ALPHA) || defined(VT_VERTEX_Y_BIAS) || defined(VT_VERTEX_ALPHA)
#define VS_INPUT_CONTAINS_TEXTURECOORD
#endif

#if defined(VT_LVERTEX) || defined(VT_TLVERTEX) || defined(VT_VERTEX_XYZCUV) || defined(VT_VERTEX_XYZC)
#define VS_INPUT_CONTAINS_COLOR
#endif

#if defined(VT_VERTEX_BUMPSKIN) || defined(VT_VERTEX_BUMP) || defined(VT_VERTEX_BUMP_Y_BIAS_ALPHA) || defined(VT_VERTEX_BUMP_Y_BIAS) || defined(VT_VERTEX_BUMP_ALPHA)
#define VS_INPUT_CONTAINS_BINORMAL
#endif

#if defined(VT_VERTEX_BUMP_Y_BIAS_ALPHA) || defined(VT_VERTEX_Y_BIAS_ALPHA) || defined(VT_VERTEX_Y_BIAS) || defined(VT_VERTEX_BUMP_Y_BIAS)
#define VS_INPUT_CONTAINS_Y_BIAS
#endif

#if defined(VT_VERTEX_BUMP_Y_BIAS_ALPHA) || defined(VT_VERTEX_Y_BIAS_ALPHA) || defined(VT_VERTEX_ALPHA) || defined(VT_VERTEX_BUMP_ALPHA)
#define VS_INPUT_CONTAINS_ALPHA_FACTOR
#endif

//////////////////////////////////////////////////////////////
// VS_OUTPUT
#if defined(SHD_FRESNEL_DIFFUSE) || defined(SHD_FRESNEL_SPECULAR) || defined(SHD_FRESNEL_ENV)
#define PS_USES_FRESNEL
#endif

#if defined(PS_USES_FRESNEL) || defined(SHD_ENV) || defined(SHD_SPECULAR)
#define PS_NEEDS_EYE_DIR
#endif

#if !defined(SHD_BUMP) && !defined(SHD_VERTICAL_NORMAL) && !defined(SHD_PRELIT)
#define PS_NEEDS_VERTEX_NORMAL
#endif

#if defined(SHD_BUMP) && !defined(SHD_VERTICAL_NORMAL) && defined(VS_INPUT_CONTAINS_BINORMAL)
#define PS_NEEDS_TANSPACE
#endif

struct VS_OUTPUT
{
    float4 vPosition    : SV_POSITION;
    float4 cDiffuse     : COLOR;
    float2 TexBase      : TEXCOORD0;

#if defined(SHD_DETAIL)
    float2 TexDetail    : TEXCOORD1;
#endif

#if !defined(SHD_PRELIT)
    #if defined(SHD_BUMP)
        float2 TexBump  : TEXCOORD2;
    #endif

    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
        float3 vNormalWS  : TEXCOORD3;
    #endif

    #ifdef PS_NEEDS_TANSPACE
        float3 vBinormalWS : TEXCOORD6;
        float3 vTangentWS  : TEXCOORD7;
    #endif

    #ifdef PS_NEEDS_EYE_DIR
        float3 vPositionWS      : TEXCOORD4;
        float3 vEyePositionWS   : TEXCOORD5;
    #endif
#endif

#if defined(SHD_SHADOWMAP)
    float4 TexSM    : TEXCOORD8;
#endif

    float  fFogDistance   : FOG;
};

#if defined(VS_INPUT_CONTAINS_NORMAL)
//
// Helper function for accumulating light values
float3 AccumulatePointLights(const float4 vPosition,
                             const float3 vNormalWS)
{
    float3 cDiffuse = float3(0,0,0);        // Lights get accumulated here

    const float3 vWorldPosition = mul(vPosition, g_mWorld);
    for(dword i = 0; i < g_iPointLightCount; i++)
    {
        const float3 vPointLightWorldSpace = g_vPointLightWorldSpace[i];
        const float dist = distance(vWorldPosition, vPointLightWorldSpace);
        if(dist < g_vPointLightRangeAttenuation[i].x)
        {
            const float3 vPointLightDirection = normalize(vPointLightWorldSpace - vWorldPosition);
            // all our point lights have attenuation coeffs set to (1,0,0) so we can simplify the attenuation equation
            const float attenuation = 1;
            float fDot = 0;
            #if defined(SHD_VERTICAL_NORMAL)
                fDot = max(vPointLightDirection.y, 0);
            #else
                fDot = saturate(dot(vNormalWS, vPointLightDirection));
            #endif

            cDiffuse += saturate(g_vPointLightDiffuse[i] * attenuation * fDot);
        }
    }
    return cDiffuse;
}
#endif

VS_OUTPUT VS(VS_INPUT In)
{
    VS_OUTPUT Out = (VS_OUTPUT) 0;

    /////////////////////////////////////////////////////////////////////////////
    // Process the Vertex Position
    float4 vPosition = In.vPosition;
    // Apply Skinning to the position if present.
    #if defined(VS_INPUT_CONTAINS_SKINNING)
        vPosition = SkinPosition(vPosition,
                                 In.vBlendWeight,
                                 g_rgmBones[In.vBlendIndices.x],
                                 g_rgmBones[In.vBlendIndices.y],
                                 g_rgmBones[In.vBlendIndices.z],
                                 g_rgmBones[In.vBlendIndices.w]);
    #endif

    #if VT_VERTEX_MATRIX
    vPosition = mul(vPosition, In.mTransform);
    #endif

    // Vertical bias from stream
    #if defined(VS_INPUT_CONTAINS_Y_BIAS) && defined(SHD_APPLY_Y_BIAS)
    {
        vPosition.y += In.vYBias.x;
    }
    #endif

    #if VT_TLVERTEX
        Out.vPosition = mul(float4(vPosition.xyz, 1.0f), g_mDeviceToViewport);
    #else
        Out.vPosition = mul(vPosition, g_mWorldViewProjection);
    #endif

    #if defined(SHD_PRELIT)
        #if defined(SHD_PRELIT_CONSTANT)
            Out.cDiffuse = float4(1,1,1,1);
        #else
            Out.cDiffuse = In.cColor.bgra; // input is RGBA - swizzle to RGBA
        #endif
        #if !defined(SHD_BASE)
            Out.cDiffuse *= g_fDiffuse;
        #endif
    ////////////////////////////////////////////////////////////////////////////
    // Process the Normal (if present)
    // Use the Normal to calculate the color
    #elif defined(VS_INPUT_CONTAINS_NORMAL)
    float3 vNormal = In.vNormal;
    {
        #if defined(VS_INPUT_CONTAINS_SKINNING)
            vNormal = SkinNormal(vNormal,
                                 In.vBlendWeight,
                                 g_rgmBones[In.vBlendIndices.x],
                                 g_rgmBones[In.vBlendIndices.y],
                                 g_rgmBones[In.vBlendIndices.z],
                                 g_rgmBones[In.vBlendIndices.w]);
        #endif

        const float3 vNormalWS = TransformVectorToSpace(vNormal, g_mWorld);

        #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
            Out.vNormalWS = vNormalWS;
        #endif

        // If we're doing bump mapping, there's no need to compute the color in the output
        // since we're going to be doing it based on the bump texture in the pixel shader
        #if defined(SHD_BUMP)
            Out.cDiffuse = float4(0,0,0,0);
        #else
        {
            #if defined(SHD_VERTICAL_NORMAL)
                const float fDot = max(g_vSunVectorWorld.y, 0);
            #else
                const float fDot = saturate(dot(vNormalWS, g_vSunVectorWorld));
            #endif

            Out.cDiffuse = float4(saturate(g_vSunAmbient + (g_vSunDirectional * fDot)), 1.0);

            #if !defined(SHD_BASE)
                Out.cDiffuse *= g_fDiffuse;
            #endif
        }
        #endif

        Out.cDiffuse += float4(AccumulatePointLights(vPosition, vNormalWS), 0.0f);
    }
    #elif defined(VS_INPUT_CONTAINS_COLOR)
        Out.cDiffuse = In.cColor.bgra; // input is RGBA - swizzle to RGBA
    #else
        Out.cDiffuse = float4(1,1,1,1);
    #endif

    ///////////////////////////////////////////////////////////////////////
    // Process Binormals & Tangents
    #if defined(PS_NEEDS_TANSPACE)
    {
        #if defined(VS_INPUT_CONTAINS_SKINNING)
            const float3 vTangent = SkinNormal(In.vTangent,
                                               In.vBlendWeight,
                                               g_rgmBones[In.vBlendIndices.x],
                                               g_rgmBones[In.vBlendIndices.y],
                                               g_rgmBones[In.vBlendIndices.z],
                                               g_rgmBones[In.vBlendIndices.w]);
        #else
            const float3 vTangent = In.vTangent;
        #endif

        Out.vTangentWS = TransformVectorToSpace(vTangent, g_mWorld);
        Out.vBinormalWS = cross(Out.vNormalWS, Out.vTangentWS);
    }
    #endif

    ////////////////////////////////////////////////////////////////////////
    // Pass Values to Pixel Shader
    #if defined(VS_INPUT_CONTAINS_TEXTURECOORD)
        Out.TexBase = In.Tex;
    #else
        Out.TexBase = float2(0,0);
    #endif

    #if defined(SHD_DETAIL)
        Out.TexDetail.xy = In.Tex * g_fDetailScale;
    #endif

    #if defined(PS_NEEDS_EYE_DIR)
        Out.vPositionWS = mul(vPosition, g_mWorld);
        Out.vEyePositionWS = g_vEyePoint;
    #endif

    #if defined(SHD_BUMP)
        Out.TexBump.xy = In.Tex * g_fBumpScale;
    #endif

    #if defined(SHD_SHADOWMAP)
        // Base texture coordinates
        float4 ShadowPos = mul(mul(vPosition,g_mWorld)+ g_vShadowOffset, g_mShadow);
        Out.TexSM.xy = ShadowPos.xy;

        float4 ShadowViewPos = mul(mul(vPosition,g_mWorld)+ g_vShadowOffset, g_mShadowView);
        Out.TexSM.z = ShadowViewPos.z / g_fShadowRange;

        // Output shadow blending factor (based on light vector. This is so that faces away
        // from the light do not get shadow applied to them
        #ifdef SHD_PRELIT
        {
            Out.TexSM.w = 1.0f;
        }
        #else
        {
            float fDot;
            #if !defined(VS_INPUT_CONTAINS_NORMAL) || defined(SHD_VERTICAL_NORMAL)
            {
                fDot = max(g_vSunVectorWorld.y, 0);
            }
            #else
            {
                fDot = saturate(dot(TransformVectorToSpace(vNormal, g_mWorld), g_vSunVectorWorld));
            }
            #endif
            Out.TexSM.w = fDot;
        }
        #endif // !SHD_PRELIT
    #endif

    // Alpha blend from stream
    #if defined(VS_INPUT_CONTAINS_ALPHA_FACTOR) && defined(SHD_APPLY_VERTEX_ALPHA_FACTOR)
    {
        o.ToPix.cDiffuse.a *= v.vAlphaFactor.x;
    }
    #endif

    // Compute fog
    Out.fFogDistance = FogVS(vPosition, g_mWorld, g_vEyePoint);

    return Out;
}

float4 PS(VS_OUTPUT In) : SV_TARGET
{
    float4 cBase;

    #if defined(SHD_BASE)
        cBase = txBase.Sample(samDefaultOverrides, In.TexBase);
    #else
        cBase = float4(1,1,1,1);
    #endif

    #ifdef SHD_DETAIL
    {
        const float3 cDetail = txDetail.Sample(samDefaultOverrides, In.TexDetail);
        cBase *= float4(2 * cDetail, 1);
    }
    #endif

    //
    // Build all terms
    //
    float4 cColor, cDiffuse;

    // Need to have the specular color here in case the alpha is needed to modulate the diffuse
    #if defined(SHD_SPECULARMAP) || defined(SHD_BLENDENVBYSPECULARMAPALPHA)
        const float4 cSpecularMap = txSpecular.Sample(samDefaultOverrides, In.TexBase);
    #endif

    #if defined(SHD_PRELIT)
        #if defined(SHD_PRELIT_CONSTANT)
            cDiffuse = cBase;
        #else
            cDiffuse = cBase * In.cDiffuse;
        #endif

        cColor = cDiffuse;

        #if defined(SHD_BLENDENVBYINVBASEALPHA)
            cDiffuse.a = g_fDiffuse.a;
        #endif
    #else
    {
        float3 vNormalWS, vVertexNormalWS;
        // Get required normals
        vVertexNormalWS = vNormalWS = float3(0, 1, 0);

        #if defined(SHD_BUMP)
        {
            const float3 vNormalSurface = SampleBump(In.TexBump);

            #if !defined(SHD_VERTICAL_NORMAL)
                vVertexNormalWS = normalize(In.vNormalWS);

                // Transform surface normal into world space
                vNormalWS = TransformVectorByColumnMatrix(vNormalSurface, normalize(In.vBinormalWS), -normalize(In.vTangentWS), vVertexNormalWS);
            #else
                vNormalWS = TransformVectorToSpace(float3(vNormalSurface.r, vNormalSurface.b, -vNormalSurface.g), g_mWorld);
            #endif
        }
        #elif !defined(SHD_VERTICAL_NORMAL) && (defined(PS_USES_FRESNEL) || defined(SHD_SPECULAR) || defined(SHD_ENV))
        {
            const float3 vNorm = normalize(In.vNormalWS);

            #if defined(PS_USES_FRESNEL)
                vVertexNormalWS = vNorm;
            #endif

            #if defined(SHD_SPECULAR) || defined(SHD_ENV)
                vNormalWS = vNorm;
            #endif
        }
        #endif // !SHD_BUMP

        #if defined(PS_NEEDS_EYE_DIR)
            const float3 vEyeDirWS = normalize(In.vEyePositionWS - In.vPositionWS);
        #endif

        //
        // Diffuse term
        //
        #if defined(SHD_BUMP)
        {
            // Perform diffuse calculation
            const float fDot = saturate(dot(vNormalWS, g_vSunVectorWorld));
            cDiffuse = cBase * float4( saturate(g_vSunAmbient + (g_vSunDirectional * fDot) ), 1) + In.cDiffuse;
        }
        #else
            // Not bumpmapping, so diffuse comes from the vertex
            cDiffuse = cBase * In.cDiffuse;
        #endif

        #if defined(SHD_BLENDDIFFUSEVBYBASEALPHA)
            cDiffuse *= cBase.a;
        #elif defined(SHD_BLENDDIFFUSEBYINVSPECULARMAPALPHA)
            cDiffuse *= (1 - cSpecularMap.a);
        #endif

        #if defined(PS_USES_FRESNEL)
            //
            // Fresnel term
            //

            // We will not bump the fresnel term as it is cheaper
            // not to and avoids sparklies
            const float4 cFresnel = CalculateFresnel(vEyeDirWS, vVertexNormalWS);
        #endif

        #if defined(SHD_FRESNEL_DIFFUSE)
            cDiffuse *= 1 - cFresnel;
        #endif

        #if defined(SHD_BLENDENVBYINVBASEALPHA)
            cDiffuse.a = g_fDiffuse.a;
        #endif

        cColor = cDiffuse;

        #if defined(SHD_SPECULAR)
        {
            //
            // Specular term
            //
            // Specular:  S = N.H ^ Power
            float3 cSpecular;
            #if defined(SHD_SPECULARMAP)
                cSpecular = CalculateSpecularFromMap10(vEyeDirWS, g_vSunVectorWorld, vNormalWS, cSpecularMap, g_fSpecularMapPowerScale, g_vSunDirectional);
            #else
                cSpecular = CalculateSpecular10(vEyeDirWS, g_vSunVectorWorld, vNormalWS, g_cSpecular.a, g_vSunDirectional, g_cSpecular);
            #endif

            #if defined(SHD_FRESNEL_SPECULAR)
                cSpecular *= cFresnel;
            #endif

            cColor += float4(cSpecular, 0);
        }
        #endif

        #if defined(SHD_ENV)
        {
            //
            // Environment term
            //
            float3 cReflection = CalculateEnv(vEyeDirWS, vNormalWS, g_vSunDirectional);

            #if defined(SHD_BLENDENVBYINVBASEALPHA)
                cReflection *= (1 - cBase.a);
            #elif defined(SHD_BLENDENVBYSPECULARMAPALPHA)
                cReflection *= cSpecularMap.a;
            #endif

            #if defined(SHD_FRESNEL_ENV)
                cReflection *= cFresnel;
            #endif

            #if defined(SHD_PRECIP_BLEND)
            {
                #if defined(SHD_PRECIP_OFFSET)
                {
                    cReflection = lerp(cReflection * g_fMaterialPrecipOffset, cReflection, g_fMaterialPrecipLevel);
                }
                #else
                {
                    cReflection *= g_fMaterialPrecipLevel;
                }
                #endif
            }
            #endif

            cColor += float4(cReflection, 0) * g_fEnvironmentLevelScale;
        }
        #endif
    }
    #endif // PRELIT

    #if defined( SHD_ADDITIVE_EMISSIVEMAP ) || defined( SHD_BLENDED_EMISSIVEMAP ) || defined( SHD_MULTIPLY_EMISSIVEMAP )
    {
        const float4 cEmissive = txEmissive.Sample(samDefaultOverrides, In.TexBase);

        float fEmissiveScale = 1.0;
        #if defined(SHD_BLOOM) && defined(SHD_ALLOW_EMISSIVE_BLOOM)
            fEmissiveScale = fHdrEmissiveScale * cEmissive.a;
        #endif
    
        #if defined( SHD_ADDITIVE_EMISSIVEMAP )
            cColor += float4(fEmissiveScale * cEmissive.rgb, 0);
        #elif defined( SHD_BLENDED_EMISSIVEMAP )
            cColor = lerp( float4(fEmissiveScale * cEmissive.rgb,0), cColor, g_fBlendedEmissiveFactor);
        #elif defined( SHD_MULTIPLY_EMISSIVEMAP )
        {
            const float fDayNightRatio = 1 - g_fBlendedEmissiveFactor;     // 1 is night, 0 is day
            const float4 cEmissiveMapped = (cDiffuse + float4(fDayNightRatio * fEmissiveScale * cEmissive.rgb,0)) * cBase;

            cColor = float4(cEmissiveMapped.rgb, cColor.a);
        }
        #endif
    }
    #endif

    #if defined(SHD_BLOOM_COPY) 
        cColor.rgb = fHdrCopyScale * cColor.rgb;
    #endif
    #if defined(SHD_BLOOM_MODALPHA) 
        cColor.rgb = fHdrModAlphaScale * cBase.a * cColor.rgb;
    #endif

    #if defined(SHD_FINAL_ALPHA_BLEND)
        cColor.a *= g_fFinalAlphaBlendFactor;
    #endif

    #if defined(SHD_ALPHA_TEST)
    {
        if (cColor.a < g_fAlphaTestThreshold)
        {
            discard;
        }
    }
    #endif

    #if defined(SHD_SHADOWMAP)
        float2 baseCoord = (((float2(1.0,-1.0)*In.TexSM.xy)+1.0f)/2.0f) * g_fShadowSize;
        float2 fFrac = float2(frac(baseCoord.x),frac(baseCoord.y));
        baseCoord.x = ((round(baseCoord.x-0.5f))-0.0f)/g_fShadowSize;
        baseCoord.y = ((round(baseCoord.y-0.5f))+0.0f)/g_fShadowSize;

        float fDistanceFromSceneCamera = In.TexSM.z + g_fShadowBias;

        // Sample the current depth sample
        float fDepth = (fDistanceFromSceneCamera > txShadowMap.Sample(shadowMapSampler, baseCoord).x);

        // We sample the neighbor samples that form a cross (up, down, left, right) and lerp them together based on the
        // sample fraction that is determined from the position of the curent pixel within the sample
        float4 Depth = fDistanceFromSceneCamera > float4(
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(-1.0f/g_fShadowSize,0.0f/g_fShadowSize)).x,
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(1.0f/g_fShadowSize,0.0f/g_fShadowSize)).x,
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(0.0f/g_fShadowSize,-1.0f/g_fShadowSize)).x,
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(0.0f/g_fShadowSize,1.0f/g_fShadowSize)).x);
        fDepth += lerp( Depth.x, Depth.y, fFrac.x) + lerp( Depth.z, Depth.w, fFrac.y);

        float2 fDepthLerp;
        float4 Depth2 = fDistanceFromSceneCamera > float4(
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(-1.0f/g_fShadowSize,-1.0f/g_fShadowSize)).x,
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(-1.0f/g_fShadowSize,1.0f/g_fShadowSize)).x,
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(1.0f/g_fShadowSize,-1.0f/g_fShadowSize)).x,
                            txShadowMap.Sample(shadowMapSampler, baseCoord + float2(1.0f/g_fShadowSize,1.0f/g_fShadowSize)).x);
        fDepthLerp.xy = lerp( Depth2.xy, Depth2.zw, fFrac.x);
        fDepth += lerp( fDepthLerp.x, fDepthLerp.y, fFrac.y);

        // Finish off by factoring the "kick-in factor"
        fDepth = saturate(g_fShadowKickIn * 0.1 * fDepth);

        cColor.xyz = cColor.xyz * (1-saturate(min(fDepth,g_fShadowIntensity)*In.TexSM.w));

        // Debug/Test code to quickly output the shadow result as a simple color
        //return float4(fDepth*fBlend,0,0,1);
    #endif

    cColor = float4(FogPS(cColor.xyz, In.fFogDistance, g_fFogDensity, g_cFog), cColor.a);

    return cColor;
}

technique10 T0
{
    pass P0
    {
        SetVertexShader(CompileShader(vs_4_0, VS()));
        SetGeometryShader(NULL);
        SetPixelShader(CompileShader(ps_4_0, PS()));
    }
}

