//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
//
// Global shader.  Handles all cases
//

#include <Common.fxh>
#include <MaterialDecl.fxh>
#include <D3D9Texture.fxh>
#include <FuncLibrary.fxh>


string XFile = "sjlcube.x";

#if defined(SHD_FRESNEL_DIFFUSE) || defined(SHD_FRESNEL_SPECULAR) || defined(SHD_FRESNEL_ENV)
#define PS_USES_FRESNEL true
#endif

#if defined(PS_USES_FRESNEL) || defined(SHD_ENV) || defined(SHD_SPECULAR)
#define PS_NEEDS_EYE_DIR true
#endif

#if !defined(SHD_BUMP) && !defined(SHD_VERTICAL_NORMAL) && !defined(SHD_PRELIT)
#define PS_NEEDS_VERTEX_NORMAL true
#endif

#if defined(SHD_BUMP) && !defined(SHD_VERTICAL_NORMAL)
#define PS_NEEDS_TANSPACE
#endif

#if defined(SHD_BASE) || defined(SHD_DETAIL) || defined(SHD_SPECULARMAP) || defined(SHD_BLENDENVBYSPECULARMAPALPHA) || defined(SHD_ADDITIVE_EMISSIVEMAP) || defined(SHD_BLENDED_EMISSIVEMAP) || defined(SHD_MULTIPLY_EMISSIVEMAP) || defined(SHD_ALLOW_EMISSIVE_BLOOM)
#define PS_NEEDS_BASE_TEXCOORD
#endif

const bool EffectDeclaration
<
    string Name      = "General";
    string Class     = "Basic";
    string Features  = "Bump";
    string Fallback  = "Assert";
    bool   NewMaterialUsage = true;
> = true;

#define vLightDir vSunVectorWorld

// Declare inputs
struct VS_INPUT
{
    float4 vPos       : POSITION;

#if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormal    : NORMAL;
#endif

#if defined(SHD_PRELIT) && !defined(SHD_PRELIT_CONSTANT)
    float4 vDiffuse   : COLOR0;
#endif

#if defined(PS_NEEDS_BASE_TEXCOORD) || (!defined(SHD_PRELIT) && defined(SHD_BUMP))
    float2 vTex       : TEXCOORD0;
#endif

#ifdef PS_NEEDS_TANSPACE
    float3 vBinormal  : BINORMAL;
    float3 vTangent   : TANGENT;
#endif

#if defined(SHD_SKIN)
    float4 vBlendWeight   : BLENDWEIGHT;
    float4 vBlendIndices  : BLENDINDICES;
#endif

#if defined(SHD_APPLY_Y_BIAS)
    float1 vYBias        : TEXCOORD2;
#endif

#if defined(SHD_APPLY_VERTEX_ALPHA_FACTOR)
    float1 vAlphaFactor  : TEXCOORD3;
#endif
};

// Declare outputs
struct PS_INPUT
{
#ifdef PS_NEEDS_BASE_TEXCOORD
    float2 TexBase  : TEXCOORD0;
#endif

#ifdef SHD_DETAIL
    float2 TexDetail  : TEXCOORD1;
#endif

    float4 cDiffuse   : COLOR0;

#ifndef SHD_PRELIT
    #ifdef SHD_BUMP
        float2 TexBump  : TEXCOORD2;
    #endif
    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
        float3 vNormalWS  : TEXCOORD3;
    #endif
    #ifdef PS_NEEDS_EYE_DIR
        float3 vPosWS     : TEXCOORD4;
        float3 vEyePosWS  : TEXCOORD5;
    #endif

    #ifdef PS_NEEDS_TANSPACE
        float3 vBinormalWS : TEXCOORD6;
        float3 vTangentWS  : TEXCOORD7;
    #endif
#endif

};

// Declare outputs
struct VS_OUTPUT
{
    float4 vPos     : POSITION;
    float  fFog     : FOG;

    PS_INPUT ToPix;
};

struct VS_OUTPUT_SM
{
    float4 vPos     : POSITION;
    float  fFog     : FOG;
    float3 TexBase  : TEXCOORD0;
    float  fBlend   : TEXCOORD1;
};

// Code
VS_OUTPUT
VS(const VS_INPUT v)
{
    VS_OUTPUT o = (VS_OUTPUT)0;
    float4 vPos = v.vPos;

    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormal = v.vNormal;
    #endif

    #ifdef SHD_SKIN

    // Convert D3DCOLOR to UBYTE4. We can't use UBYTE4 types directly because
    // most 1.x shader hardware does not support it. As a sollution, we use
    // D3DCOLOR and use the D3DCOLORtoUBYTE4 macro to tweak the values back.
    float4 vBlendWeight = D3DCOLORtoUBYTE4(v.vBlendWeight) / 255.0f;
    float4 vBlendIndices = D3DCOLORtoUBYTE4(v.vBlendIndices);

    // Apply skinning on position
    vPos = SkinPosition(vPos,
                        vBlendWeight,
                        mBones[vBlendIndices.x],
                        mBones[vBlendIndices.y],
                        mBones[vBlendIndices.z],
                        mBones[vBlendIndices.w]);

    // Apply skinning on normal (if needed)
    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    vNormal = SkinNormal(vNormal,
                         vBlendWeight,
                         mBones[vBlendIndices.x],
                         mBones[vBlendIndices.y],
                         mBones[vBlendIndices.z],
                         mBones[vBlendIndices.w]);
    #endif //normal

    #endif //skin

    // Vertical bias from stream
    #ifdef SHD_APPLY_Y_BIAS
    {
        vPos.y += v.vYBias.x;
    }
    #endif

    // Transform to clip space
    o.vPos = mul(vPos, mFullProj);

    #ifdef SHD_BASE
    // Base texture coordinates
    o.ToPix.TexBase.xy = v.vTex;
    #endif

    #ifdef SHD_DETAIL
    {
        o.ToPix.TexDetail.xy = v.vTex * DetailScale;
    }
    #endif

    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormalWS = TransformVectorToSpace(vNormal, mWorld);
    o.ToPix.vNormalWS = vNormalWS;
    #endif

    #ifdef SHD_PRELIT
    {
        #ifdef SHD_PRELIT_CONSTANT
            o.ToPix.cDiffuse = float4(1,1,1,1);
        #else
            o.ToPix.cDiffuse = v.vDiffuse;
        #endif

        #ifndef SHD_BASE
        {
            o.ToPix.cDiffuse *= MatDiffuse;
        }
        #endif // !SHD_BASE
    }
    #else
        #ifdef SHD_BUMP
        {
            o.ToPix.TexBump.xy = v.vTex * BumpScale;
            #ifdef PS_NEEDS_TANSPACE
            {
                #ifdef SHD_SKIN
                float3 vTangent = SkinNormal(v.vTangent,
                                             vBlendWeight,
                                             mBones[vBlendIndices.x],
                                             mBones[vBlendIndices.y],
                                             mBones[vBlendIndices.z],
                                             mBones[vBlendIndices.w]);
                #else
                float3 vTangent  = v.vTangent;
                #endif

                // Transform tangent into worldspace and calculate binormal
                o.ToPix.vTangentWS  = TransformVectorToSpace(vTangent,  mWorld);
                o.ToPix.vBinormalWS = cross(vNormalWS, o.ToPix.vTangentWS);
            }
            #endif
            // clear the diffuse because the point lights will accumulate here
            o.ToPix.cDiffuse = float4(0,0,0,0);
        }
        #else // !SHD_BUMP
        {
            float fDot = 0;

            #ifdef SHD_VERTICAL_NORMAL
            {
                fDot = max(vLightDir.y, 0);
            }
            #else
            {
                #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
                {
                    fDot = saturate(dot(vNormalWS, vLightDir));
                }
                #endif
            }
            #endif

            o.ToPix.cDiffuse = float4( saturate( vSunAmbient + (vSunDirectional * fDot) ), 1.0);

            #ifndef SHD_BASE
            {
                o.ToPix.cDiffuse *= MatDiffuse;
            }
            #endif // !SHD_BASE
        }
        #endif // !SHD_BUMP
    #endif // !SHD_PRELIT

    #ifndef SHD_PRELIT
    {
        // calculate point lights
        float3 vWorldPos = mul(vPos, mWorld);
        for(dword i=0; i < iPointLightCount; i++)
        {
            float dist = distance(vWorldPos, vPointLightWorldSpace[i]);
            if (dist < vPointLightRangeAttenuation[i].x)
            {
                float3 vPointLightDir = normalize(vPointLightWorldSpace[i] - vWorldPos);
                // all our point lights have attenuation coeffs set to (1,0,0) so we can simplify the attenuation equation
                float attenuation = 1;
                float fDot = 0;
                #ifdef SHD_VERTICAL_NORMAL
                {
                    fDot = max(vPointLightDir.y, 0);
                }
                #else
                {
                    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
                    {
                        fDot = saturate(dot(vNormalWS, vPointLightDir));
                    }
                    #endif
                }
                #endif
                o.ToPix.cDiffuse += float4(saturate(vPointLightDiffuse[i] * attenuation * fDot), 0.0f);
            }
        }

        #ifdef PS_NEEDS_EYE_DIR
        {
            o.ToPix.vPosWS    = mul(vPos, mWorld);
            o.ToPix.vEyePosWS = vEyePoint;
        }
        #endif
    }
    #endif

    // Alpha blend from stream
    #ifdef	SHD_APPLY_VERTEX_ALPHA_FACTOR
    {
	o.ToPix.cDiffuse.a *= v.vAlphaFactor.x;
    }
    #endif

    // Compute fog
    o.fFog = ComputeFog(o.vPos.w, FOG_END, FOG_RECIP, FOG_SELECT_VERTEX, FOG_SELECT_TABLE);

    return o;
}

VS_OUTPUT_SM
VS_ShadowMap(const VS_INPUT v)
{
    VS_OUTPUT_SM o = (VS_OUTPUT_SM)0;

    float4 vPos = v.vPos;

    #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
    float3 vNormal = v.vNormal;
    #endif

    #ifdef SHD_SKIN
    {
        // Convert D3DCOLOR to UBYTE4. We can't use UBYTE4 types directly because
        // most 1.x shader hardware does not support it. As a sollution, we use
        // D3DCOLOR and use the D3DCOLORtoUBYTE4 macro to tweak the values back.
        float4 vBlendWeight = D3DCOLORtoUBYTE4(v.vBlendWeight) / 255.0f;
        float4 vBlendIndices = D3DCOLORtoUBYTE4(v.vBlendIndices);

        // Apply skinning on position
        vPos = SkinPosition(vPos,
                            vBlendWeight,
                            mBones[vBlendIndices.x],
                            mBones[vBlendIndices.y],
                            mBones[vBlendIndices.z],
                            mBones[vBlendIndices.w]);

        #if defined(PS_NEEDS_VERTEX_NORMAL) || defined(PS_NEEDS_TANSPACE)
        // Apply skinning on normal
        vNormal = SkinNormal(vNormal,
                             vBlendWeight,
                             mBones[vBlendIndices.x],
                             mBones[vBlendIndices.y],
                             mBones[vBlendIndices.z],
                             mBones[vBlendIndices.w]);
        #endif
    }
    #endif //skin

    // Transform to clip space
    o.vPos = mul(vPos, mFullProj);

    #ifdef SHD_PRELIT
    {
        o.fBlend = 1.0f;
    }
    #else
    {
        float fDot;
        #ifdef SHD_VERTICAL_NORMAL
        {
            fDot = max(vLightDir.y, 0);
        }
        #else
        {
            fDot = saturate(dot(TransformVectorToSpace(vNormal, mWorld), vLightDir));
        }
        #endif
        o.fBlend = fDot;
    }
    #endif // !SHD_PRELIT

    // Base texture coordinates
    float4 ShadowPos = mul(mul(vPos,mWorld)+vShadowOffset, mShadowMtx);
    o.TexBase.xy = ShadowPos.xy;

    float4 ShadowViewPos = mul(mul(vPos,mWorld)+vShadowOffset, mShadowViewMtx);
    o.TexBase.z = ShadowViewPos.z / fShadowRange;

    return o;
}

float4
PS(PS_INPUT Input) : COLOR
{
    //
    // Base texture and detail
    //
    float4 cBase;
    #ifdef SHD_BASE
    {
        cBase = tex2D(BaseSampler, Input.TexBase);
    }
    #else
    {
        cBase = float4(1, 1, 1, 1);
    }
    #endif

    #ifdef SHD_DETAIL
    {
        float3 cDetail = tex2D(DetailSampler, Input.TexDetail);
        cBase *= float4(2 * cDetail, 1);
    }
    #endif

    //
    // Build all terms
    //
    float4 cColor, cDiffuse, cFresnel;

    // Need to have the specular color here in case the alpha is needed to modulate the diffuse
    #if defined(SHD_SPECULARMAP) || defined(SHD_BLENDENVBYSPECULARMAPALPHA)
    float4 cSpecularMap = tex2D(SpecularSampler, Input.TexBase);
    #endif

    float3 vEyeDirWS;
    float3 vNormalWS, vVtxNormalWS;


    #if defined(SHD_PRELIT)
    {
        #if defined(SHD_PRELIT_CONSTANT)
            cDiffuse = cBase;
        #else
            cDiffuse = cBase * Input.cDiffuse;
        #endif
        cColor = cDiffuse;

        #if defined(SHD_BLENDENVBYINVBASEALPHA)
        {
            cDiffuse.a = MatDiffuse.a;
        }
        #endif
    }
    #else
    {
        // Get required normals
        vVtxNormalWS = vNormalWS = float3(0, 1, 0);

        #if defined(SHD_BUMP)
        {
            float3 vNormalSurface = SampleBump(Input.TexBump);

            #if !defined(SHD_VERTICAL_NORMAL)
            {
                vVtxNormalWS = normalize(Input.vNormalWS);

                // Transform surface normal into world space
                vNormalWS = TransformVectorByColumnMatrix(vNormalSurface, normalize(Input.vBinormalWS), -normalize(Input.vTangentWS), vVtxNormalWS);
            }
            #else
            {
                vNormalWS = TransformVectorToSpace(float3(vNormalSurface.r, vNormalSurface.b, -vNormalSurface.g), mWorld);
            }
            #endif
        }
        #else  // !SHD_BUMP
        {
            #if !defined(SHD_VERTICAL_NORMAL) && (defined(PS_USES_FRESNEL) || defined(SHD_SPECULAR) || defined(SHD_ENV))
            {
                float3 vNorm = normalize(Input.vNormalWS);

                #if defined(PS_USES_FRESNEL)
                {
                    vVtxNormalWS = vNorm;
                }
                #endif

                #if defined(SHD_SPECULAR) || defined(SHD_ENV)
                {
                    vNormalWS = vNorm;
                }
                #endif
            }
            #endif
        }
        #endif // !SHD_BUMP

        #if defined(PS_NEEDS_EYE_DIR)
        {
            vEyeDirWS = normalize(Input.vEyePosWS - Input.vPosWS);
        }
        #endif

        #if defined(PS_USES_FRESNEL)
        {
            //
            // Fresnel term
            //

            // We will not bump the fresnel term as it is cheaper
            // not to and avoids sparklies
            cFresnel = CalculateFresnel(vEyeDirWS, vVtxNormalWS);
        }
        #endif

        //
        // Diffuse term
        //
        #if defined(SHD_BUMP)
        {
            // Perform diffuse calculation
            float fDot = saturate(dot(vNormalWS, vLightDir));
            cDiffuse = cBase * float4( saturate(vSunAmbient + (vSunDirectional * fDot) ), 1) + Input.cDiffuse;
        }
        #else
        {
            // Not bumpmapping, so diffuse comes from the vertex
            cDiffuse = cBase * Input.cDiffuse;
        }
        #endif
        #if defined(SHD_BLENDDIFFUSEVBYBASEALPHA)
        {
          cDiffuse *= cBase.a;
        }
        #else
        {
            #if defined(SHD_BLENDDIFFUSEBYINVSPECULARMAPALPHA)
            {
                cDiffuse *= (1 - cSpecularMap.a);
            }
            #endif
        }
        #endif

        #if defined(SHD_FRESNEL_DIFFUSE)
        {
            cDiffuse *= 1 - cFresnel;
        }
        #endif

        #if defined(SHD_BLENDENVBYINVBASEALPHA)
        {
            cDiffuse.a = MatDiffuse.a;
        }
        #endif

        cColor = cDiffuse;

        #if defined(SHD_SPECULAR)
        {
            //
            // Specular term
            //
            // Specular:  S = N.H ^ Power
            float3 cSpecular;
            #if defined(SHD_SPECULARMAP)
            {
                cSpecular = CalculateSpecularFromMap(vEyeDirWS, vLightDir, vNormalWS, cSpecularMap, MatSpecMapPowerScale, vSunDirectional);
            }
            #else
            {
                cSpecular = CalculateSpecular(vEyeDirWS, vLightDir, vNormalWS, SpecularPower, vSunDirectional, SpecularColor);
            }
            #endif

            #if defined(SHD_FRESNEL_SPECULAR)
            {
                cSpecular *= cFresnel;
            }
            #endif

            cColor += float4(cSpecular, 0);
        }
        #endif

        #if defined(SHD_ENV)
        {
            //
            // Environment term
            //
            float3 cReflection = CalculateEnv(vEyeDirWS, vNormalWS, vSunDirectional);

            #if defined(SHD_BLENDENVBYINVBASEALPHA)
            {
                cReflection *= (1 - cBase.a);
            }
            #else
            {
                #if defined(SHD_BLENDENVBYSPECULARMAPALPHA)
                {
                    cReflection *= cSpecularMap.a;
                }
                #endif
            }
            #endif

            #if defined(SHD_FRESNEL_ENV)
            {
                cReflection *= cFresnel;
            }
            #endif

            #if defined(SHD_PRECIP_BLEND)
            {
                #if defined(SHD_PRECIP_OFFSET)
                {
                    cReflection = lerp(cReflection * MatPrecipOffset, cReflection, GlobalPrecipLevel);
                }
                #else
                {
                    cReflection *= GlobalPrecipLevel;
                }
                #endif
            }
            #endif

            cColor += float4(cReflection, 0) * EnvLevelScale;
        }
        #endif
    }
    #endif // PRELIT

    #if defined( SHD_ADDITIVE_EMISSIVEMAP ) || defined( SHD_BLENDED_EMISSIVEMAP ) || defined( SHD_MULTIPLY_EMISSIVEMAP )
    {
        float4 cEmissive = tex2D(EmissiveSampler, Input.TexBase);

        #if defined( SHD_ADDITIVE_EMISSIVEMAP )
        {
            cColor += float4(cEmissive.r, cEmissive.g, cEmissive.b, 0);
        }
        #elif defined( SHD_BLENDED_EMISSIVEMAP )
        {
            cColor = lerp(cEmissive, cColor, State_BlendedEmissiveFactor);
        }
        #elif defined( SHD_MULTIPLY_EMISSIVEMAP )
        {
            float fDayNightRatio = 1 - State_BlendedEmissiveFactor;     // 1 is night, 0 is day
            float4 cEmissiveMapped = (cDiffuse + (fDayNightRatio * cEmissive)) * cBase;

            cColor = float4(cEmissiveMapped.r, cEmissiveMapped.g, cEmissiveMapped.b, cColor.a);
        }
        #endif
    }
    #endif

    #if defined(SHD_FINAL_ALPHA_BLEND)
    {
        cColor.a *= State_FinalAlphaBlendFactor;
    }
    #endif

    #if defined(SHD_BLEND_CONSTANT)
    {
        cColor *= State_BlendConstant;
    }
    #endif

    // Bloom
    #ifdef SHD_BLOOM
    {
        float fBloom = 0;

        #if defined(SHD_SPECULAR) && !defined(SHD_NOSPECULARBLOOM) && !defined(SHD_PRELIT)
        {
            float fPower;

            #if defined(SHD_SPECULARMAP)
            {
                fPower = cSpecularMap.a * MatSpecMapPowerScale;
            }
            #else
            {
                fPower = SpecularPower;
            }
            #endif

            if (fPower > 0)
            {
                #if !defined(SHD_VERTICAL_NORMAL)
                {
                    fBloom = CalculateSpecularBloom(vEyeDirWS, vLightDir, vNormalWS, fPower, MatSpecBloomFloor);
                }
                #else
                {
                    fBloom = CalculateSpecularBloom(vEyeDirWS, vLightDir, fPower, MatSpecBloomFloor);
                }
                #endif
            }
        }
        #else
        {
            #if defined(SHD_BLOOM_COPY) || defined(SHD_BLOOM_MODALPHA)
            {
                #if defined(SHD_BLOOM_MODALPHA)
                {
                    fBloom = cBase.a;
                }
                #else
                {
                    fBloom = 1;
                }
                #endif
            }
            #endif

        }
        #endif


        // Add in emissive bloom if it's active
        {
            #if defined(SHD_ALLOW_EMISSIVE_BLOOM)
            {
                float fEmissiveBloomAlpha = tex2D(EmissiveSampler, Input.TexBase).a;


            fBloom += fEmissiveBloomAlpha;
            }
            #endif
        }

        cColor.rgb *= fBloom;
    }
    #endif

    return (cColor);
}

float4
PS_ShadowMap(float3 TexBase:TEXCOORD0, float fBlend:TEXCOORD1) : COLOR
{
    //
    // Base texture and detail
    //

    float2 baseCoord = (((float2(1.0,-1.0)*TexBase)+1.0f)/2.0f) * 2048.0f;
    float2 fFrac = float2(frac(baseCoord.x),frac(baseCoord.y));
    baseCoord.x = ((round(baseCoord.x-0.5f))-0.0f)/2048.0f;
    baseCoord.y = ((round(baseCoord.y-0.5f))+0.0f)/2048.0f;

    float fDistanceFromSceneCamera = TexBase.z + fShadowBias;

    // Sample the current depth sample
    float fDepth = (fDistanceFromSceneCamera > tex2D(ShadowMapSampler, baseCoord).x);

    // We sample the neighbor samples that form a cross (up, down, left, right) and lerp them together based on the
    // sample fraction that is determined from the position of the curent pixel within the sample
    float4 Depth = fDistanceFromSceneCamera > float4(
                        tex2D(ShadowMapSampler, baseCoord + float2(-1.0f/2048.0f,0.0f/2048.0f)).x,
                        tex2D(ShadowMapSampler, baseCoord + float2(1.0f/2048.0f,0.0f/2048.0f)).x,
                        tex2D(ShadowMapSampler, baseCoord + float2(0.0f/2048.0f,-1.0f/2048.0f)).x,
                        tex2D(ShadowMapSampler, baseCoord + float2(0.0f/2048.0f,1.0f/2048.0f)).x);
    fDepth += lerp( Depth.x, Depth.y, fFrac.x) + lerp( Depth.z, Depth.w, fFrac.y);

    #ifdef SHD_SHADOWMAP_HIGH
        // For the high quality filter, we also sample the corner samples to get a full 3x3 filter
        float2 fDepthLerp;
        float4 Depth2 = fDistanceFromSceneCamera > float4(
                            tex2D(ShadowMapSampler, baseCoord + float2(-1.0f/2048.0f,-1.0f/2048.0f)).x,
                            tex2D(ShadowMapSampler, baseCoord + float2(-1.0f/2048.0f,1.0f/2048.0f)).x,
                            tex2D(ShadowMapSampler, baseCoord + float2(1.0f/2048.0f,-1.0f/2048.0f)).x,
                            tex2D(ShadowMapSampler, baseCoord + float2(1.0f/2048.0f,1.0f/2048.0f)).x);
        fDepthLerp.xy = lerp( Depth2.xy, Depth2.zw, fFrac.x);
        fDepth += lerp( fDepthLerp.x, fDepthLerp.y, fFrac.y);
    #endif

    // Finish off by factoring the "kick-in factor"
    fDepth = saturate(fShadowKickIn * 0.1 * fDepth);

    return float4(0,0,0,(min(fDepth,fShadowIntensity))*fBlend);

    // Debug/Test code to quickly output the shadow result as a simple color
    //return float4(fDepth*fBlend,0,0,1);
}

technique T0
<
    int    Performance = EffectPerfLevel_MaxShader20;
>
{
    pass P0
    {
        #include <DefaultStates.fxh>

#ifdef SHD_BLOOM
        FogColor = 0;
#endif
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS();

        AlphaBlendEnable = (State_AlphaBlendEnable);
        SrcBlend         = (State_SrcBlend);
        DestBlend        = (State_DstBlend);
    }
#ifdef SHD_SHADOWMAP
    pass ShadowMap
    {
        #include <DefaultStates.fxh>

        VertexShader = compile vs_2_0 VS_ShadowMap();
        PixelShader  = compile ps_2_0 PS_ShadowMap();

        AlphaBlendEnable = (State_ShadowAlphaBlendEnable);
        SrcBlend             = SrcAlpha;
        DestBlend            = InvSrcAlpha;
    }
#endif
}
