//---------------------------------------------------------------------------
// Flight Simulator X - Shader Effect Files
// Copyright (c) 2006, Microsoft Corporation
//---------------------------------------------------------------------------
#ifndef _FUNCLIBRARY_FXH_
#define _FUNCLIBRARY_FXH_

//
// Fog Helper Functions
// FogVS will fog the vertex and output a distance that must be passed to the pixel shader
// FogPS will take a "base" color and fog it based on the linearly interpolated distance brought in from the vertex shader
float FogVS(const float4 vPosition,
            const matrix mWorld,
            const float4 vEyePoint)
{
    return distance(vEyePoint, mul(vPosition, mWorld));
}

#define E 2.71828183
float3 FogPS(const float3 cColor,
             const float  fDistance,
             const float  fFogDensity,
             const float3 cFog)
{
    float fFogCoefficient = 1.0 / pow( E, fDistance * fFogDensity );

    fFogCoefficient = clamp(fFogCoefficient, 0, 1.0);
    
    return lerp(cFog, cColor, fFogCoefficient);
}

float ComputeFog(const float w, 
                 const float fFogEnd, 
                 const float fFogRecip, 
                 const float fFogSelectVertex, 
                 const float fFogSelectTable)
{
    // Compute vertex fog
    // fog = (fog_end - dist) * (1 / (fog_end-fog_start))
    float fFog = (fFogEnd - w) * fFogRecip;
    fFog = max(fFog, 0.0);
    fFog = min(fFog, 1.0);

    // Now select whether we're using vertex fog or table fog
    fFog = (fFog * fFogSelectVertex) + (w * fFogSelectTable);

    return (fFog);
}

float3 TransformVectorToSpace(const float3 v, const matrix m)
{
    return (normalize(mul(v, m)));
}

float3 TransformVectorByColumnMatrix(const float3 vIn, const float3 column1, const float3 column2, const float3 column3)
{
    float3x3 xformMatrix = float3x3(column1.x, column2.x, column3.x,
                                    column1.y, column2.y, column3.y,
                                    column1.z, column2.z, column3.z);
    return (mul(xformMatrix, vIn));
}

float4 SkinPosition(const float4 vIn, 
                    const float4 vWeight,                     
                    const float4x3 mBone1,
                    const float4x3 mBone2,
                    const float4x3 mBone3,
                    const float4x3 mBone4)
{
    float3 res;   

    res = vWeight.x * mul(vIn,mBone1);
    res += vWeight.y * mul(vIn,mBone2);
    res += vWeight.z * mul(vIn,mBone3);
    res += vWeight.w * mul(vIn,mBone4);

    return float4(res,1);
}

float3 SkinNormal(const float3 vIn, 
                  const float4 vWeight, 
                  const float3x3 mBone1, 
                  const float3x3 mBone2,
                  const float3x3 mBone3,
                  const float3x3 mBone4)
{
    float3 res;

    res =  vWeight.x * mul(vIn, mBone1);
    res += vWeight.y * mul(vIn, mBone2);
    res += vWeight.z * mul(vIn, mBone3);
    res += vWeight.w * mul(vIn, mBone4);

    return (res);
}

#if !defined(SHD_VERTICAL_NORMAL)

float CalculateSpecularBloom(const float3 vEyeDir, 
                             const float3 vLight, 
                             const float3 vNormal, 
                             const float fPower,
                             const float fMatSpecBloomFloor)
{
    float3 vHN = normalize(vEyeDir + vLight);

	// need .0001 bias to prevent artifacts
    float fSpec = pow(saturate(dot(vNormal, vHN)) + 0.0001, 64 * fPower);

    if (fSpec < fMatSpecBloomFloor)
    {
         fSpec = 0;
    }

    return (fSpec);
}

#else

float CalculateSpecularBloom(const float3 vEyeDir, 
                             const float3 vLight, 
                             const float fPower,
                             const float fMatSpecBloomFloor)
{
    float3 vHN = normalize(vEyeDir + vLight);

#ifdef SHD_HDR
    float fSpec = pow(saturate(vHN.y + 0.0001), 64 * fPower);
#else
    float fSpec = pow(saturate(vHN.y), 64 * fPower);
#endif

    if (fSpec < fMatSpecBloomFloor)
    {
         fSpec = 0;
    }

    return (fSpec);
}

#endif

float3 CalculateSpecular(const float3 vEyeDir, 
                         const float3 vLight, 
                         const float3 vNormal,
                         const float  fSpecularPower,
                         const float3 vSunDirectional,
                         const float3 fSpecularColor)
{
    float3 vHN = normalize(vEyeDir + vLight);

#if defined(SHD_BUMP) || !defined(SHD_VERTICAL_NORMAL)
#ifdef SHD_HDR
    float fSpec = pow(saturate(dot(vNormal, vHN)) + 0.0001, fSpecularPower);
#else
    float fSpec = pow(saturate(dot(vNormal, vHN)), fSpecularPower);
#endif
#else
    float fSpec = pow(saturate(vHN.y), fSpecularPower);
#endif

    float3 cSpecular = fSpec * fSpecularColor * vSunDirectional;

    return (cSpecular);
}

float3 CalculateSpecularFromMap(const float3 vEyeDir, 
                                const float3 vLight, 
                                const float3 vNormal, 
                                const float4 cSpecularMap,
                                const float  fMatSpecMapPowerScale,
                                const float3 vSunDirectional)
{
    float3 vHN = normalize(vEyeDir + vLight);
#ifdef SHD_HDR
    float  fPow = (cSpecularMap.a * fMatSpecMapPowerScale) + 0.0001;
#else
    // biasing specular power scale
    float  fPow = cSpecularMap.a * fMatSpecMapPowerScale + 0.0001;
#endif

#if defined(SHD_BUMP) || !defined(SHD_VERTICAL_NORMAL)
    float fSpec = pow(saturate(dot(vNormal, vHN)), fPow);
#else
    float fSpec = pow(saturate(vHN.y), fPow);
#endif

    float3 cSpecular = fSpec * cSpecularMap * vSunDirectional;

    return (cSpecular);
}

#ifdef SAMPLERS_DEFINED

float4 CalculateFresnel(float3 vEyeDir, float3 vNormal)
{
    // Fresnel:   F = Tex(N.E);
    float fFresnel;

#ifdef SHD_VERTICAL_NORMAL
    fFresnel = 1 - vEyeDir.y;
#else
    #ifdef SHD_DOUBLE_SIDED
        fFresnel = 1 - abs( dot(vNormal, vEyeDir) );
    #else
        fFresnel = 1 - dot(vNormal, vEyeDir);
    #endif
#endif

    return SampleFresnelTexture(fFresnel);
}

float3 CalculateEnv(const float3 vEyeDir, 
                    const float3 vNormal,
                    const float3 vSunDirectional)
{
    // Glance:    G = 2 * (V.N) * N - V
#if defined(SHD_BUMP) || !defined(SHD_VERTICAL_NORMAL)
    float3 vGlance = reflect(vEyeDir, vNormal);
#else
    float3 vGlance = float3(0, 2 * vEyeDir.y, 0) - vEyeDir;
#endif

    // Environment cube map coordinates
    float3 cReflection = vSunDirectional * SampleEnvironmentTexture(float3(vGlance.x, -vGlance.y, vGlance.z));

    return cReflection;
}

// Bumpmap helpers
float3 SampleBump(float2 vTex)
{    
    float3 vSample = (SampleBumpTexture(vTex).agb * 2) - 1;

    return normalize(float3(vSample.r, vSample.g, vSample.b));
}

#endif // SAMPLERS_DEFINED

#endif // _FUNCLIBRARY_FXH_
